;;; sendfax.el -- fax sending commands for GNU Emacs.
;;;
;;; Copyright (C) 1995 Ralph Schleicher
;;;
;;; This library is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU Library General Public
;;; License as published by the Free Software Foundation; either
;;; version 2 of the License, or (at your option) any later version.
;;;
;;; This library is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Library General Public License for more details.
;;;
;;; You should have received a copy of the GNU Library General Public
;;; License along with this library; if not, write to the Free
;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; This file is not part of GNU Emacs.
;;;
;;; Author: Ralph Schleicher <rs@purple.IN-Ulm.DE>
;;; Maintainer: see the `Author' field
;;; Keywords: mail
;;; Comments: The code of the fax-send-it function is derived from the
;;; sendmail-send-it function defined in `sendmail.el' as distributed
;;; with GNU Emacs.
;;; Code:


(require 'sendmail)

;;;### autoload
(defvar sendfax-program "/usr/local/bin/faxpr" "\
*Program used to send facsimile messages.")

;;;### autoload
(defvar sendfax-switches '("-t" "mail") "\
*List of arguments when SENDFAX-PROGRAM is invoked.")

;;;### autoload
(defvar fax-archive-file-name nil "\
*Name of file to write all outgoing facsimile messages in or nil for none.")

;;;### autoload
(defvar fax-signature nil "\
*Text inserted at the end of a fax message during initialization.")

(defvar fax-mode-hook nil)
(defvar fax-send-hook nil)

(defun fax-to () "\
Move point to the end of the X-Fax-To field."
  (interactive) (expand-abbrev)
  (mail-position-on-field "X-Fax-To"))

(defun fax-cc () "\
Move point to the end of the X-Fax-CC field.  The fax-cc command creates
a X-Fax-CC field if none exists."
  (interactive) (expand-abbrev)
  (or (mail-position-on-field "X-Fax-CC" t)
      (progn (mail-position-on-field "X-Fax-To")
	     (insert "\nX-Fax-CC: "))))

(defun fax-bcc () "\
Move point to the end of the X-Fax-BCC field.  The fax-bcc command creates
a X-Fax-BCC field if none exists."
  (interactive) (expand-abbrev)
  (or (mail-position-on-field "X-Fax-BCC" t)
      (progn (mail-position-on-field "X-Fax-CC")
	     (insert "\nX-Fax-BCC: "))))

(defun fax (&optional no-erase) "\
Edit a fax message to be sent via a Mail-to-Fax gateway."
  (interactive "P")
  (let ((mail-default-headers fax-default-headers)
	(mail-archive-file-name fax-archive-file-name)
	(mail-signature fax-signature))
    (mail no-erase "FAX")
    (let ((case-fold-search t))
      (goto-char (point-min))
      (if (re-search-forward "^X-Fax-Date:[ \t]*|\\(.+\\)\n" nil t)
	  (let ((command (buffer-substring (match-beginning 1) (match-end 1))))
	    (delete-region (match-beginning 1) (match-end 1))
	    (delete-backward-char 2) (shell-command command t))))
    (or (mail-position-on-field "X-Fax-To" t)
	(progn (mail-position-on-field "Subject")
	       (insert "\nX-Fax-To: ")))
    (make-local-variable 'send-mail-function)
    (setq send-mail-function 'fax-send-it)
    (make-local-variable 'mail-abbrev-mode-regexp)
    (setq mail-abbrev-mode-regexp "^X-Fax-\\(To\\|CC\\|BCC\\):")
    (make-local-variable 'mail-send-hook)
    (setq mail-send-hook fax-send-hook)
    (mail-position-on-field "Subject")
    (run-hooks 'fax-mode-hook)))

(defun fax-send () "\
Send this buffer as a fax message."
  (goto-char (point-min))
  (if (re-search-forward
       (concat "^" (regexp-quote mail-header-separator) "\n") nil t)
      nil
    (if (search-forward "\n\n" nil t)
	nil
      (goto-char (point-max))
      (if (= (current-column) 0)
	  (insert "\n")
	(insert "\n\n")))
    (backward-char 1)
    (insert mail-header-separator))
  (set-buffer-modified-p t)
  (let ((buffer-file-name nil)
	(mail-interactive nil))
    (make-local-variable 'send-mail-function)
    (setq send-mail-function 'fax-send-it)
    (mail-send)))

(defun fax-send-it () "\
Almost the same as sendmail-send-it."
  (let ((fax-buf (generate-new-buffer " sendfax"))
	(err-buf (if mail-interactive
		     (generate-new-buffer " sendfax errors")
		   0))
	(mail-buf (current-buffer))
	(case-fold-search nil))
    (unwind-protect
	(save-excursion
	  (set-buffer fax-buf)
	  (erase-buffer)
	  (insert-buffer-substring mail-buf)
	  (goto-char (point-max))
	  (or (= (preceding-char) ?\n)
	      (insert ?\n))
	  (goto-char (point-min))
	  (re-search-forward
	   (concat "^" (regexp-quote mail-header-separator) "\n"))
	  (replace-match "\n")
	  (backward-char 1)
	  (setq delim-line (point-marker))
	  (goto-char (point-min))
	  (while (and (re-search-forward "\n\n\n*" delim-line t)
		      (< (point) delim-line))
	    (replace-match "\n"))
	  (let ((case-fold-search t))
	    (goto-char (point-min))
	    (while (re-search-forward "^[^:]+:[ \t]*\n" delim-line t)
	      (replace-match ""))
	    (goto-char (point-min))
	    (if (re-search-forward "^Subject:" delim-line t)
		(replace-match "X-Fax-Subject:"))
	    (goto-char (point-min))
	    (if (re-search-forward "^FCC:" delim-line t)
		(mail-do-fcc delim-line)))
	  (if mail-interactive
	      (save-excursion
		(set-buffer err-buf)
		(erase-buffer)))
	  (apply 'call-process-region
		 (nconc (list (point-min) (point-max)
			      sendfax-program nil err-buf nil)
			(if (stringp sendfax-switches)
			    (list sendfax-switches)
			  sendfax-switches)))
	  (if mail-interactive
	      (save-excursion
		(set-buffer err-buf)
		(if (not (zerop (buffer-size)))
		    (error "Sending...failed")))))
      (kill-buffer fax-buf)
      (if (bufferp err-buf)
	  (kill-buffer err-buf)))))

(provide 'sendfax)


;;; sendfax.el ends here
