/* b23.c --- boundary line between regions 2 and 3.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-errno.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.9: Coefficients of equations (3.22) to (3.24).  */
struct t39
  {
    /* Index.  */
    int i;

    /* Coefficient.  */
    double n;
  };

static const struct t39 t39[] =
  {
    {1,  0.34805185628969E+03},
    {2, -0.11671859879975E+01},
    {3,  0.10192970039326E-02},
    {4,  0.57254459862746E+03},
    {5,  0.13918839778870E+02},
  };

#define n(i) t39[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the boundary pressure corresponding to the boundary
   temperature T.  */
double
if97i_p_b23 (double t)
{
  return (n(0) + t / t_star * (n(1) + t / t_star * n(2))) * p_star;
}


/* Return the boundary temperature corresponding to the boundary
   pressure P.  */
double
if97i_t_b23 (double p)
{
  return (n(3) + sqrt ((p / p_star - n(4)) / n(2))) * t_star;
}


/* Return the boundary pressure corresponding to the boundary
   temperature T.

   This function signals a domain error if the temperature T is
   not in the range [623.15 863.15] K.  */
double
if97_p_b23 (double t)
{
  if (t < 623.15 || t > 863.15)
    set_errno (EDOM);

  return if97i_p_b23 (t);
}


/* Return the boundary temperature corresponding to the boundary
   pressure P.

   This function signals a domain error if the pressure P is not
   in the range [16.5292E+6 100.0E+6] Pa.  */
double
if97_t_b23 (double p)
{
  if (p < 16.5292E+6 || p > 100.0E+6)
    set_errno (EDOM);

  return if97i_t_b23 (p);
}
