/* h1.c --- equations for region 1 (enthalpy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.5: Coefficients and exponents of equation (3.4).  */
struct t35
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t35 t35[] =
  {
    { 1, 0,  0, -0.23872489924521E+03},
    { 2, 0,  1,  0.40421188637945E+03},
    { 3, 0,  2,  0.11349746881718E+03},
    { 4, 0,  6, -0.58457616048039E+01},
    { 5, 0, 22, -0.15285482413140E-03},
    { 6, 0, 32, -0.10866707695377E-05},
    { 7, 1,  0, -0.13391744872602E+02},
    { 8, 1,  1,  0.43211039183559E+02},
    { 9, 1,  2, -0.54010067170506E+02},
    {10, 1,  3,  0.30535892203916E+02},
    {11, 1,  4, -0.65964749423638E+01},
    {12, 1, 10,  0.93965400878363E-02},
    {13, 1, 32,  0.11573647505340E-06},
    {14, 2, 10, -0.25858641282073E-04},
    {15, 2, 32, -0.40644363084799E-08},
    {16, 3, 10,  0.66456186191635E-07},
    {17, 3, 32,  0.80670734103027E-10},
    {18, 4, 32, -0.93477771213947E-12},
    {19, 5, 32,  0.58265442020601E-14},
    {20, 6, 32, -0.15020185953503E-16},
  };

#define I(i) t35[i].I
#define J(i) t35[i].J
#define n(i) t35[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference enthalpy.  */
#define h_star 2.5E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_h1_p_star (void)
{
  return p_star;
}


/* Return the reference enthalpy.  */
double
if97i_h1_h_star (void)
{
  return h_star;
}


/* Return the reference temperature.  */
double
if97i_h1_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_h1_pi (double p)
{
  return p / p_star;
}


/* Return the reduced enthalpy.  */
double
if97i_h1_eta (double h)
{
  return h / h_star;
}


/* Return the reduced temperature when using equation (3.4).  */
double
if97i_h1_theta (double pi, double eta)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 20; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (eta + 1.0), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and enthalpy H.  */
double
if97i_h1_t (double p, double h)
{
  double pi, eta, theta;

  pi = if97i_h1_pi (p);
  eta = if97i_h1_eta (h);

  theta = if97i_h1_theta (pi, eta);

  return theta * t_star;
}
