/* h2a.c --- equations for subregion 2a (enthalpy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.20: Coefficients and exponents of equation (3.15).  */
struct t320
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t320 t320[] =
  {
    { 1, 0,  0,  0.10898952318288E+04},
    { 2, 0,  1,  0.84951654495535E+03},
    { 3, 0,  2, -0.10781748091826E+03},
    { 4, 0,  3,  0.33153654801263E+02},
    { 5, 0,  7, -0.74232016790248E+01},
    { 6, 0, 20,  0.11765048724356E+02},
    { 7, 1,  0,  0.18445749355790E+01},
    { 8, 1,  1, -0.41792700549624E+01},
    { 9, 1,  2,  0.62478196935812E+01},
    {10, 1,  3, -0.17344563108114E+02},
    {11, 1,  7, -0.20058176862096E+03},
    {12, 1,  9,  0.27196065473796E+03},
    {13, 1, 11, -0.45511318285818E+03},
    {14, 1, 18,  0.30919688604755E+04},
    {15, 1, 44,  0.25226640357872E+06},
    {16, 2,  0, -0.61707422868339E-02},
    {17, 2,  2, -0.31078046629583E+00},
    {18, 2,  7,  0.11670873077107E+02},
    {19, 2, 36,  0.12812798404046E+09},
    {20, 2, 38, -0.98554909623276E+09},
    {21, 2, 40,  0.28224546973002E+10},
    {22, 2, 42, -0.35948971410703E+10},
    {23, 2, 44,  0.17227349913197E+10},
    {24, 3, 24, -0.13551334240775E+05},
    {25, 3, 44,  0.12848734664650E+08},
    {26, 4, 12,  0.13865724283226E+01},
    {27, 4, 32,  0.23598832556514E+06},
    {28, 4, 44, -0.13105236545054E+08},
    {29, 5, 32,  0.73999835474766E+04},
    {30, 5, 36, -0.55196697030060E+06},
    {31, 5, 42,  0.37154085996233E+07},
    {32, 6, 34,  0.19127729239660E+05},
    {33, 6, 44, -0.41535164835634E+06},
    {34, 7, 28, -0.62459855192507E+02},
  };

#define I(i) t320[i].I
#define J(i) t320[i].J
#define n(i) t320[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference enthalpy.  */
#define h_star 2.0E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_h2a_p_star (void)
{
  return p_star;
}


/* Return the reference enthalpy.  */
double
if97i_h2a_h_star (void)
{
  return h_star;
}


/* Return the reference temperature.  */
double
if97i_h2a_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_h2a_pi (double p)
{
  return p / p_star;
}


/* Return the reduced enthalpy.  */
double
if97i_h2a_eta (double h)
{
  return h / h_star;
}


/* Return the reduced temperature.  */
double
if97i_h2a_theta (double pi, double eta)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (eta - 2.1), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and enthalpy H.  */
double
if97i_h2a_t (double p, double h)
{
  double pi, eta, theta;

  pi = if97i_h2a_pi (p);
  eta = if97i_h2a_eta (h);

  theta = if97i_h2a_theta (pi, eta);

  return theta * t_star;
}
