/* h2b.c --- equations for subregion 2b (enthalpy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.20: Coefficients and exponents of equation (3.15).  */
struct t321
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t321 t321[] =
  {
    { 1, 0,  0,  0.14895041079516E+04},
    { 2, 0,  1,  0.74307798314034E+03},
    { 3, 0,  2, -0.97708318797837E+02},
    { 4, 0, 12,  0.24742464705674E+01},
    { 5, 0, 18, -0.63281320016026E+00},
    { 6, 0, 24,  0.11385952129658E+01},
    { 7, 0, 28, -0.47811863648625E+00},
    { 8, 0, 40,  0.85208123431544E-02},
    { 9, 1,  0,  0.93747147377932E+00},
    {10, 1,  2,  0.33593118604916E+01},
    {11, 1,  6,  0.33809355601454E+01},
    {12, 1, 12,  0.16844539671904E+00},
    {13, 1, 18,  0.73875745236695E+00},
    {14, 1, 24, -0.47128737436186E+00},
    {15, 1, 28,  0.15020273139707E+00},
    {16, 1, 40, -0.21764114219750E-02},
    {17, 2,  2, -0.21810755324761E-01},
    {18, 2,  8, -0.10829784403677E+00},
    {19, 2, 18, -0.46333324635812E-01},
    {20, 2, 40,  0.71280351959551E-04},
    {21, 3,  1,  0.11032831789999E-03},
    {22, 3,  2,  0.18955248387902E-03},
    {23, 3, 12,  0.30891541160537E-02},
    {24, 3, 24,  0.13555504554949E-02},
    {25, 4,  2,  0.28640237477456E-06},
    {26, 4, 12, -0.10779857357512E-04},
    {27, 4, 18, -0.76462712454814E-04},
    {28, 4, 24,  0.14052392818316E-04},
    {29, 4, 28, -0.31083814331434E-04},
    {30, 4, 40, -0.10302738212103E-05},
    {31, 5, 18,  0.28217281635040E-06},
    {32, 5, 24,  0.12704902271945E-05},
    {33, 5, 40,  0.73803353468292E-07},
    {34, 6, 28, -0.11030139238909E-07},
    {35, 7,  2, -0.81456365207833E-13},
    {36, 7, 28, -0.25180545682962E-10},
    {37, 9,  1, -0.17565233969407E-17},
    {38, 9, 40,  0.86934156344163E-14},
  };

#define I(i) t321[i].I
#define J(i) t321[i].J
#define n(i) t321[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference enthalpy.  */
#define h_star 2.0E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_h2b_p_star (void)
{
  return p_star;
}


/* Return the reference enthalpy.  */
double
if97i_h2b_h_star (void)
{
  return h_star;
}


/* Return the reference temperature.  */
double
if97i_h2b_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_h2b_pi (double p)
{
  return p / p_star;
}


/* Return the reduced enthalpy.  */
double
if97i_h2b_eta (double h)
{
  return h / h_star;
}


/* Return the reduced temperature.  */
double
if97i_h2b_theta (double pi, double eta)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 38; ++i)
    {
      double tem = n(i);

      prod (tem, (pi - 2.0), I(i));
      prod (tem, (eta - 2.6), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and enthalpy H.  */
double
if97i_h2b_t (double p, double h)
{
  double pi, eta, theta;

  pi = if97i_h2b_pi (p);
  eta = if97i_h2b_eta (h);

  theta = if97i_h2b_theta (pi, eta);

  return theta * t_star;
}
