/* h2c.c --- equations for subregion 2c (enthalpy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.20: Coefficients and exponents of equation (3.15).  */
struct t322
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t322 t322[] =
  {
    { 1, -7,  0, -0.32368398555242E+13},
    { 2, -7,  4,  0.73263350902181E+13},
    { 3, -6,  0,  0.35825089945447E+12},
    { 4, -6,  2, -0.58340131851590E+12},
    { 5, -5,  0, -0.10783068217470E+11},
    { 6, -5,  2,  0.20825544563171E+11},
    { 7, -2,  0,  0.61074783564516E+06},
    { 8, -2,  1,  0.85977722535580E+06},
    { 9, -1,  0, -0.25745723604170E+05},
    {10, -1,  2,  0.31081088422714E+05},
    {11,  0,  0,  0.12082315865936E+04},
    {12,  0,  1,  0.48219755109255E+03},
    {13,  1,  4,  0.37966001272486E+01},
    {14,  1,  8, -0.10842984880077E+02},
    {15,  2,  4, -0.45364172676660E-01},
    {16,  6,  0,  0.14559115658698E-12},
    {17,  6,  1,  0.11261597407230E-11},
    {18,  6,  4, -0.17804982240686E-10},
    {19,  6, 10,  0.12324579690832E-06},
    {20,  6, 12, -0.11606921130984E-05},
    {21,  6, 16,  0.27846367088554E-04},
    {22,  6, 20, -0.59270038474176E-03},
    {23,  6, 22,  0.12918582991878E-02},
  };

#define I(i) t322[i].I
#define J(i) t322[i].J
#define n(i) t322[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference enthalpy.  */
#define h_star 2.0E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_h2c_p_star (void)
{
  return p_star;
}


/* Return the reference enthalpy.  */
double
if97i_h2c_h_star (void)
{
  return h_star;
}


/* Return the reference temperature.  */
double
if97i_h2c_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_h2c_pi (double p)
{
  return p / p_star;
}


/* Return the reduced enthalpy.  */
double
if97i_h2c_eta (double h)
{
  return h / h_star;
}


/* Return the reduced temperature.  */
double
if97i_h2c_theta (double pi, double eta)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 23; ++i)
    {
      double tem = n(i);

      prod (tem, (pi + 25.0), I(i));
      prod (tem, (eta - 1.8), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature of water at pressure P and enthalpy H.  */
double
if97i_h2c_t (double p, double h)
{
  double pi, eta, theta;

  pi = if97i_h2c_pi (p);
  eta = if97i_h2c_eta (h);

  theta = if97i_h2c_theta (pi, eta);

  return theta * t_star;
}
