/* if97.h --- IAPWS-IF97 declarations.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#ifndef IAPWS_IF97_H
#define IAPWS_IF97_H


#if ! defined (__GNUC__) || (__GNUC__ < 2)
#ifndef __attribute__
#define __attribute__(arg)
#endif /* not __attribute__ */
#endif /* not __GNUC__ or __GNUC__ < 2 */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * IF97 data structure.
 */

struct if97
  {
    /* Pressure in Pa.  */
    double if_p;

    /* Temperature in K.  */
    double if_t;

    /* Specific volume in m^3/kg.  */
    double if_v;

    /* Specific enthalpy in J/kg.  */
    double if_h;

    /* Specific internal energy in J/kg.  */
    double if_u;

    /* Specific entropy in J/kg/K.  */
    double if_s;

    /* Specific isobaric heat capacity in J/kg/K.  */
    double if_cp;

    /* Specific isochoric heat capacity in J/kg/K.  */
    double if_cv;

    /* Speed of sound in m/s.  */
    double if_w;
  };

/* Allocate an array of IF97 data structures with N elements.  */
extern struct if97 *if97_create (size_t __n);

/* Free the array of IF97 data structure pointed at by MEM.  */
extern struct if97 *if97_destroy (struct if97 *__mem);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Reference constants.
 */

/* Specific gas constant of water.  */
#define IF97_R 461.526

/* Molar gas constant.  */
#define IF97_RM 8.31451

/* Molar mass of water.  */
#define IF97_MM 18.015257E-3

/* Critical temperature of water.  */
#define IF97_TC 647.096

/* Critical pressure of water.  */
#define IF97_PC 22.064E+6

/* Critical density of water.  */
#define IF97_RHOC 322.0

/* Triple-point temperature of water.  */
#define IF97_TT 273.16

/* Triple-point pressure of water.  */
#define IF97_PT 611.657

/* Boiling-point temperature of water at 101325 Pa.  */
#define IF97_TB 373.1243

/* Return the specific gas constant of water.  */
extern double if97_r (void) __attribute__ ((const));

/* Return the molar gas constant.  */
extern double if97_r_m (void) __attribute__ ((const));

/* Return the molar mass of water.  */
extern double if97_m_m (void) __attribute__ ((const));

/* Return the critical temperature of water.  */
extern double if97_t_c (void) __attribute__ ((const));

/* Return the critical pressure of water.  */
extern double if97_p_c (void) __attribute__ ((const));

/* Return the critical density of water.  */
extern double if97_rho_c (void) __attribute__ ((const));

/* Return the triple-point temperature of water.  */
extern double if97_t_t (void) __attribute__ ((const));

/* Return the triple-point pressure of water.  */
extern double if97_p_t (void) __attribute__ ((const));

/* Return the boiling-point temperature of water at 101325 Pa.  */
extern double if97_t_b (void) __attribute__ ((const));

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Saturation line.
 */

/* Return the saturation pressure corresponding to the saturation
   temperature T.  */
extern double if97_p_s (double __t) __attribute__ ((const));

/* Return the saturation temperature corresponding to the saturation
   pressure P.  */
extern double if97_t_s (double __p) __attribute__ ((const));

/* Return non-zero if pressure P and temperature T define saturation
   conditions inside the uncertainties of the equations.  */
extern int if97_saturation_p (double __p, double __t) __attribute__ ((const));

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Boundary line between regions 2 and 3.
 */

/* Return the boundary pressure corresponding to the boundary
   temperature T.  */
extern double if97_p_b23 (double __t) __attribute__ ((const));

/* Return the boundary temperature corresponding to the boundary
   pressure P.  */
extern double if97_t_b23 (double __p) __attribute__ ((const));

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Temperature interface.
 */

/* Calculate all properties of water at pressure P and temperature T.  */
extern struct if97 *if97t (struct if97 *__prop, double __p, double __t);

/* Return the specific volume of water in m^3/kg at pressure P and
   temperature T.  */
extern double if97t_v (double __p, double __t) __attribute__ ((const));

/* Return the specific enthalpy of water in J/kg at pressure P and
   temperature T.  */
extern double if97t_h (double __p, double __t) __attribute__ ((const));

/* Return the specific internal energy of water in J/kg at pressure P
   and temperature T.  */
extern double if97t_u (double __p, double __t) __attribute__ ((const));

/* Return the specific entropy of water in J/kg at pressure P and
   temperature T.  */
extern double if97t_s (double __p, double __t) __attribute__ ((const));

/* Return the specific isobaric heat capacity of water in J/kg/K at
   pressure P and temperature T.  */
extern double if97t_cp (double __p, double __t) __attribute__ ((const));

/* Return the specific isochoric heat capacity of water in J/kg/K at
   pressure P and temperature T.  */
extern double if97t_cv (double __p, double __t) __attribute__ ((const));

/* Return the speed of sound of water in m/s at pressure P and
   temperature T.  */
extern double if97t_w (double __p, double __t) __attribute__ ((const));

/* Return the IAPWS-IF97 region corresponding to pressure P and
   temperature T.  Value is either 1, 2, 3, or 5.  The saturation
   line (region 4) is never considered by this function.

   The following precedence rules apply if the arguments are on the
   boundary of two regions:

        * 1-2 boundary: region 1
        * 1-3 boundary: region 1
        * 3-2 boundary: region 3
        * 2-5 boundary: region 2

   A value of -1 is returned if the pressure P or the temperature T
   is outside the argument range.  */
extern int if97t_region (double __p, double __t) __attribute__ ((const));

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Enthalpy interface.
 */

/* Calculate all properties of water at pressure P and specific
   enthalpy H.

   The result is returned in the variable PROP.  If PROP is a null
   pointer, then the storage space for PROP will be allocated using
   'if97_create'.

   Return value is the address of PROP, or a null pointer if an error
   occurs.  */
extern struct if97 *if97h (struct if97 *__prop, double __p, double __h);

/* Return the temperature of water in K at pressure P and specific
   enthalpy H.  */
extern double if97h_t (double __p, double __h) __attribute__ ((const));

/* Return the specific volume of water in m^3/kg at pressure P and
   specific enthalpy H.  */
extern double if97h_v (double __p, double __h) __attribute__ ((const));

/* Return the specific internal energy of water in J/kg at pressure P
   and specific enthalpy H.  */
extern double if97h_u (double __p, double __h) __attribute__ ((const));

/* Return the specific entropy of water in J/kg at pressure P and
   specific enthalpy H.  */
extern double if97h_s (double __p, double __h) __attribute__ ((const));

/* Return the specific isobaric heat capacity of water in J/kg/K at
   pressure P and specific enthalpy H.  */
extern double if97h_cp (double __p, double __h) __attribute__ ((const));

/* Return the specific isochoric heat capacity of water in J/kg/K at
   pressure P and specific enthalpy H.  */
extern double if97h_cv (double __p, double __h) __attribute__ ((const));

/* Return the speed of sound of water in m/s at pressure P and specific
   enthalpy H.  */
extern double if97h_w (double __p, double __h) __attribute__ ((const));

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Entropy interface.
 */

/* Calculate all properties of water at pressure P and specific
   entropy S.

   The result is returned in the variable PROP.  If PROP is a null
   pointer, then the storage space for PROP will be allocated using
   'malloc'.

   Return value is the address of PROP, or a null pointer if an error
   occurs.  */
extern struct if97 *if97s (struct if97 *__prop, double __p, double __s);

/* Return the temperature of water in K at pressure P and specific
   entropy S.  */
extern double if97s_t (double __p, double __s) __attribute__ ((const));

/* Return the specific volume of water in m^3/kg at pressure P and
   specific entropy S.  */
extern double if97s_v (double __p, double __s) __attribute__ ((const));

/* Return the specific enthalpy of water in J/kg at pressure P and
   specific entropy S.  */
extern double if97s_h (double __p, double __s) __attribute__ ((const));

/* Return the specific internal energy of water in J/kg at pressure P
   and specific entropy S.  */
extern double if97s_u (double __p, double __s) __attribute__ ((const));

/* Return the specific isobaric heat capacity of water in J/kg/K at
   pressure P and specific entropy S.  */
extern double if97s_cp (double __p, double __s) __attribute__ ((const));

/* Return the specific isochoric heat capacity of water in J/kg/K at
   pressure P and specific entropy S.  */
extern double if97s_cv (double __p, double __s) __attribute__ ((const));

/* Return the speed of sound of water in m/s at pressure P and specific
   entropy S.  */
extern double if97s_w (double __p, double __s) __attribute__ ((const));


#endif /* not IAPWS_IF97_H */
