/* if97_test.c --- test procedure for the IF97 library.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#include <c-stand.h>
#include <c-math.h>
#include <string.h>
#include <stdio.h>

#include "if97.h"
#include "if97i.h"

/* Result column.  */
#define COL 40

/* Pad characters.  */
static char dots[COL + 1] = "........................................";

/* Number of tests.  */
static int tests = 0;

/* Number of failed tests.  */
static int errors = 0;


/* Report test result.  */
static int
report (const char *id, int success)
{
  ++tests;

  if (! success)
    ++errors;

  printf ("%s %.*s %s\n",
	  id, (COL - strlen (id) - 2), dots,
	  success ? "success" : "failure");

  return success;
}


/* Test if VAL is equal to REF.  */
static void
equal (const char *id, double val, double ref)
{
  report (id, (val == ref));
}


/* Test if VAL is approximately equal to REF.  */
static void
approx (const char *id, double val, double ref, double eps)
{
  if (! report (id, (fabs (val - ref) < eps)))
    {
      printf ("\tfabs (%.*G - %.*G) < %.*G\n",
	      DBL_DIG, val, DBL_DIG, ref, DBL_DIG, eps);
    }
}


/* Test procedure.  */
int
main (void)
{
  struct if97 prop;
  double p, T, h, s, rho;

  printf ("\nReference Constants");
  printf ("\n~~~~~~~~~~~~~~~~~~~\n");

  equal ("R", if97_r (), 461.526);
  equal ("R_m", if97_r_m (), 8.31451);
  equal ("M", if97_m_m (), 18.015257E-3);
  equal ("T_c", if97_t_c (), 647.096);
  equal ("p_c", if97_p_c (), 22.064E+6);
  equal ("rho_c", if97_rho_c (), 322.0);
  equal ("T_t", if97_t_t (), 273.16);
  equal ("p_t", if97_p_t (), 611.657);
  equal ("T_b", if97_t_b (), 373.1243);

  printf ("\nTemperature Interface Region 1");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.4.  */
  p = 3.0E+6;
  T = 300.0;

  if97i_t1 (&prop, p, T);
  approx ("if_v(p=3M,T=300)", prop.if_v, 0.00100215168, 0.5E-11);
  approx ("if_h(p=3M,T=300)", prop.if_h, 115331.273, 0.5E-3);
  approx ("if_u(p=3M,T=300)", prop.if_u, 112324.818, 0.5E-3);
  approx ("if_s(p=3M,T=300)", prop.if_s, 392.294792, 0.5E-6);
  approx ("if_cp(p=3M,T=300)", prop.if_cp, 4173.01218, 0.5E-5);
  approx ("if_w(p=3M,T=300)", prop.if_w, 1507.73921, 0.5E-5);

  approx ("v(p=3M,T=300)", if97i_t1_v (p, T), 0.00100215168, 0.5E-11);
  approx ("h(p=3M,T=300)", if97i_t1_h (p, T), 115331.273, 0.5E-3);
  approx ("u(p=3M,T=300)", if97i_t1_u (p, T), 112324.818, 0.5E-3);
  approx ("s(p=3M,T=300)", if97i_t1_s (p, T), 392.294792, 0.5E-6);
  approx ("cp(p=3M,T=300)", if97i_t1_cp (p, T), 4173.01218, 0.5E-5);
  approx ("w(p=3M,T=300)", if97i_t1_w (p, T), 1507.73921, 0.5E-5);

  p = 80.0E+6;
  T = 300.0;

  if97i_t1 (&prop, p, T);
  approx ("if_v(p=80M,T=300)", prop.if_v, 0.000971180894, 0.5E-12);
  approx ("if_h(p=80M,T=300)", prop.if_h, 184142.828, 0.5E-3);
  approx ("if_u(p=80M,T=300)", prop.if_u, 106448.356, 0.5E-3);
  approx ("if_s(p=80M,T=300)", prop.if_s, 368.563852, 0.5E-6);
  approx ("if_cp(p=80M,T=300)", prop.if_cp, 4010.08987, 0.5E-5);
  approx ("if_w(p=80M,T=300)", prop.if_w, 1634.69054, 0.5E-5);

  approx ("v(p=80M,T=300)", if97i_t1_v (p, T), 0.000971180894, 0.5E-12);
  approx ("h(p=80M,T=300)", if97i_t1_h (p, T), 184142.828, 0.5E-3);
  approx ("u(p=80M,T=300)", if97i_t1_u (p, T), 106448.356, 0.5E-3);
  approx ("s(p=80M,T=300)", if97i_t1_s (p, T), 368.563852, 0.5E-6);
  approx ("cp(p=80M,T=300)", if97i_t1_cp (p, T), 4010.08987, 0.5E-5);
  approx ("w(p=80M,T=300)", if97i_t1_w (p, T), 1634.69054, 0.5E-5);

  p = 3.0E+6;
  T = 500.0;

  if97i_t1 (&prop, p, T);
  approx ("if_v(p=3M,T=500)", prop.if_v, 0.00120241800, 0.5E-11);
  approx ("if_h(p=3M,T=500)", prop.if_h, 975542.239, 0.5E-3);
  approx ("if_u(p=3M,T=500)", prop.if_u, 971934.985, 0.5E-3);
  approx ("if_s(p=3M,T=500)", prop.if_s, 2580.41912, 0.5E-5);
  approx ("if_cp(p=3M,T=500)", prop.if_cp, 4655.80682, 0.5E-5);
  approx ("if_w(p=3M,T=500)", prop.if_w, 1240.71337, 0.5E-5);

  approx ("v(p=3M,T=500)", if97i_t1_v (p, T), 0.00120241800, 0.5E-11);
  approx ("h(p=3M,T=500)", if97i_t1_h (p, T), 975542.239, 0.5E-3);
  approx ("u(p=3M,T=500)", if97i_t1_u (p, T), 971934.985, 0.5E-3);
  approx ("s(p=3M,T=500)", if97i_t1_s (p, T), 2580.41912, 0.5E-5);
  approx ("cp(p=3M,T=500)", if97i_t1_cp (p, T), 4655.80682, 0.5E-5);
  approx ("w(p=3M,T=500)", if97i_t1_w (p, T), 1240.71337, 0.5E-5);

  printf ("\nEnthalpy Interface Region 1");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.6.  */
  approx ("T(p=3M,h=500k)", if97i_h1_t (3.0E+6, 500.0E+3), 391.798509, 0.5E-6);
  approx ("T(p=80M,h=500k)", if97i_h1_t (80.0E+6, 500.0E+3), 378.108626, 0.5E-6);
  approx ("T(p=80M,h=1500k)", if97i_h1_t (80.0E+6, 1500.0E+3), 611.041229, 0.5E-6);

  printf ("\nEntropy Interface Region 1");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.8.  */
  approx ("T(p=3M,s=0.5k)", if97i_s1_t (3.0E+6, 0.5E+3), 307.842258, 0.5E-6);
  approx ("T(p=80M,s=0.5k)", if97i_s1_t (80.0E+6, 0.5E+3), 309.979785, 0.5E-6);
  approx ("T(p=80M,s=3k)", if97i_s1_t (80.0E+6, 3.0E+3), 565.899909, 0.5E-6);

  printf ("\nTemperature Interface Region 2");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.15.  */
  p = 3.5E+3;
  T = 300.0;

  if97i_t2 (&prop, p, T);
  approx ("if_v(p=3.5k,T=300)", prop.if_v, 39.4913866, 0.5E-7);
  approx ("if_h(p=3.5k,T=300)", prop.if_h, 2549911.45, 0.5E-2);
  approx ("if_u(p=3.5k,T=300)", prop.if_u, 2411691.60, 0.5E-2);
  approx ("if_s(p=3.5k,T=300)", prop.if_s, 8522.38967, 0.5E-5);
  approx ("if_cp(p=3.5k,T=300)", prop.if_cp, 1913.00162, 0.5E-5);
  approx ("if_w(p=3.5k,T=300)", prop.if_w, 427.920172, 0.5E-6);

  approx ("v(p=3.5k,T=300)", if97i_t2_v (p, T), 39.4913866, 0.5E-7);
  approx ("h(p=3.5k,T=300)", if97i_t2_h (p, T), 2549911.45, 0.5E-2);
  approx ("u(p=3.5k,T=300)", if97i_t2_u (p, T), 2411691.60, 0.5E-2);
  approx ("s(p=3.5k,T=300)", if97i_t2_s (p, T), 8522.38967, 0.5E-5);
  approx ("cp(p=3.5k,T=300)", if97i_t2_cp (p, T), 1913.00162, 0.5E-5);
  approx ("w(p=3.5k,T=300)", if97i_t2_w (p, T), 427.920172, 0.5E-6);

  p = 3.5E+3;
  T = 700.0;

  if97i_t2 (&prop, p, T);
  approx ("if_v(p=3.5k,T=700)", prop.if_v, 92.3015898, 0.5E-7);
  approx ("if_h(p=3.5k,T=700)", prop.if_h, 3335683.75, 0.5E-2);
  approx ("if_u(p=3.5k,T=700)", prop.if_u, 3012628.19, 0.5E-2);
  approx ("if_s(p=3.5k,T=700)", prop.if_s, 10174.9996, 0.5E-4);
  approx ("if_cp(p=3.5k,T=700)", prop.if_cp, 2081.41274, 0.5E-5);
  approx ("if_w(p=3.5k,T=700)", prop.if_w, 644.289068, 0.5E-6);

  approx ("v(p=3.5k,T=700)", if97i_t2_v (p, T), 92.3015898, 0.5E-7);
  approx ("h(p=3.5k,T=700)", if97i_t2_h (p, T), 3335683.75, 0.5E-2);
  approx ("u(p=3.5k,T=700)", if97i_t2_u (p, T), 3012628.19, 0.5E-2);
  approx ("s(p=3.5k,T=700)", if97i_t2_s (p, T), 10174.9996, 0.5E-4);
  approx ("cp(p=3.5k,T=700)", if97i_t2_cp (p, T), 2081.41274, 0.5E-5);
  approx ("w(p=3.5k,T=700)", if97i_t2_w (p, T), 644.289068, 0.5E-6);

  p = 30.0E+6;
  T = 700.0;

  if97i_t2 (&prop, p, T);
  approx ("if_v(p=30M,T=700)", prop.if_v, 0.00542946619, 0.5E-11);
  approx ("if_h(p=30M,T=700)", prop.if_h, 2631494.74, 0.5E-2);
  approx ("if_u(p=30M,T=700)", prop.if_u, 2468610.76, 0.5E-2);
  approx ("if_s(p=30M,T=700)", prop.if_s, 5175.40298, 0.5E-5);
  approx ("if_cp(p=30M,T=700)", prop.if_cp, 10350.5092, 0.5E-4);
  approx ("if_w(p=30M,T=700)", prop.if_w, 480.386523, 0.5E-6);

  approx ("v(p=30M,T=700)", if97i_t2_v (p, T), 0.00542946619, 0.5E-11);
  approx ("h(p=30M,T=700)", if97i_t2_h (p, T), 2631494.74, 0.5E-2);
  approx ("u(p=30M,T=700)", if97i_t2_u (p, T), 2468610.76, 0.5E-2);
  approx ("s(p=30M,T=700)", if97i_t2_s (p, T), 5175.40298, 0.5E-5);
  approx ("cp(p=30M,T=700)", if97i_t2_cp (p, T), 10350.5092, 0.5E-4);
  approx ("w(p=30M,T=700)", if97i_t2_w (p, T), 480.386523, 0.5E-6);

  printf ("\nEnthalpy Interface Region 2");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.24.  */
  approx ("T(p=1k,h=3000k)", if97i_h2a_t (1.0E+3, 3000.0E+3), 534.433241, 0.5E-6);
  approx ("T(p=3M,h=3000k)", if97i_h2a_t (3.0E+6, 3000.0E+3), 575.373370, 0.5E-6);
  approx ("T(p=3M,h=4000k)", if97i_h2a_t (3.0E+6, 4000.0E+3), 1010.77577, 0.5E-5);

  approx ("T(p=5M,h=3500k)", if97i_h2b_t (5.0E+6, 3500.0E+3), 801.299102, 0.5E-6);
  approx ("T(p=5M,h=4000k)", if97i_h2b_t (5.0E+6, 4000.0E+3), 1015.31583, 0.5E-5);
  approx ("T(p=25M,h=3500k)", if97i_h2b_t (25.0E+6, 3500.0E+3), 875.279054, 0.5E-6);

  approx ("T(p=40M,h=2700k)", if97i_h2c_t (40.0E+6, 2700.0E+3), 743.056411, 0.5E-6);
  approx ("T(p=60M,h=2700k)", if97i_h2c_t (60.0E+6, 2700.0E+3), 791.137067, 0.5E-6);
  approx ("T(p=60M,h=3200k)", if97i_h2c_t (60.0E+6, 3200.0E+3), 882.756860, 0.5E-6);

  approx ("T(p=1k,h=3000k)", if97i_h2_t (1.0E+3, 3000.0E+3), 534.433241, 0.5E-6);
  approx ("T(p=3M,h=3000k)", if97i_h2_t (3.0E+6, 3000.0E+3), 575.373370, 0.5E-6);
  approx ("T(p=3M,h=4000k)", if97i_h2_t (3.0E+6, 4000.0E+3), 1010.77577, 0.5E-5);

  approx ("T(p=5M,h=3500k)", if97i_h2_t (5.0E+6, 3500.0E+3), 801.299102, 0.5E-6);
  approx ("T(p=5M,h=4000k)", if97i_h2_t (5.0E+6, 4000.0E+3), 1015.31583, 0.5E-5);
  approx ("T(p=25M,h=3500k)", if97i_h2_t (25.0E+6, 3500.0E+3), 875.279054, 0.5E-6);

  approx ("T(p=40M,h=2700k)", if97i_h2_t (40.0E+6, 2700.0E+3), 743.056411, 0.5E-6);
  approx ("T(p=60M,h=2700k)", if97i_h2_t (60.0E+6, 2700.0E+3), 791.137067, 0.5E-6);
  approx ("T(p=60M,h=3200k)", if97i_h2_t (60.0E+6, 3200.0E+3), 882.756860, 0.5E-6);

  printf ("\nEntropy Interface Region 2");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.29.  */
  approx ("T(p=0.1M,s=7.5k)", if97i_s2a_t (0.1E+6, 7.5E+3), 399.517097, 0.5E-6);
  approx ("T(p=0.1M,s=8.0k)", if97i_s2a_t (0.1E+6, 8.0E+3), 514.127081, 0.5E-6);
  approx ("T(p=2.5M,s=8.0k)", if97i_s2a_t (2.5E+6, 8.0E+3), 1039.84917, 0.5E-5);

  approx ("T(p=8M,s=6.0k)", if97i_s2b_t (8.0E+6, 6.0E+3), 600.484040, 0.5E-6);
  approx ("T(p=8M,s=7.5k)", if97i_s2b_t (8.0E+6, 7.5E+3), 1064.95556, 0.5E-5);
  approx ("T(p=90M,s=6.0k)", if97i_s2b_t (90.0E+6, 6.0E+3), 1038.01126, 0.5E-5);

  approx ("T(p=20M,s=5.75k)", if97i_s2c_t (20.0E+6, 5.75E+3), 697.992849, 0.5E-6);
  approx ("T(p=80M,s=5.25k)", if97i_s2c_t (80.0E+6, 5.25E+3), 854.011484, 0.5E-6);
  approx ("T(p=80M,s=5.75k)", if97i_s2c_t (80.0E+6, 5.75E+3), 949.017998, 0.5E-6);

  approx ("T(p=0.1M,s=7.5k)", if97i_s2_t (0.1E+6, 7.5E+3), 399.517097, 0.5E-6);
  approx ("T(p=0.1M,s=8.0k)", if97i_s2_t (0.1E+6, 8.0E+3), 514.127081, 0.5E-6);
  approx ("T(p=2.5M,s=8.0k)", if97i_s2_t (2.5E+6, 8.0E+3), 1039.84917, 0.5E-5);

  approx ("T(p=8M,s=6.0k)", if97i_s2_t (8.0E+6, 6.0E+3), 600.484040, 0.5E-6);
  approx ("T(p=8M,s=7.5k)", if97i_s2_t (8.0E+6, 7.5E+3), 1064.95556, 0.5E-5);
  approx ("T(p=90M,s=6.0k)", if97i_s2_t (90.0E+6, 6.0E+3), 1038.01126, 0.5E-5);

  approx ("T(p=20M,s=5.75k)", if97i_s2_t (20.0E+6, 5.75E+3), 697.992849, 0.5E-6);
  approx ("T(p=80M,s=5.25k)", if97i_s2_t (80.0E+6, 5.25E+3), 854.011484, 0.5E-6);
  approx ("T(p=80M,s=5.75k)", if97i_s2_t (80.0E+6, 5.75E+3), 949.017998, 0.5E-6);

  printf ("\nTemperature Interface Region 3");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  /* Table 3.33.  */
  rho = 500.0;
  T = 650.0;

  if97i_t3 (&prop, rho, T);
  approx ("if_p(rho=500,T=650)", prop.if_p, 25583701.8, 0.5E-1);
  approx ("if_h(rho=500,T=650)", prop.if_h, 1863430.19, 0.5E-2);
  approx ("if_u(rho=500,T=650)", prop.if_u, 1812262.79, 0.5E-2);
  approx ("if_s(rho=500,T=650)", prop.if_s, 4054.27273, 0.5E-5);
  approx ("if_cp(rho=500,T=650)", prop.if_cp, 13893.5717, 0.5E-4);
  approx ("if_w(rho=500,T=650)", prop.if_w, 502.005554, 0.5E-6);

  approx ("p(rho=500,T=650)", if97i_t3_p (rho, T), 25583701.8, 0.5E-1);
  approx ("h(rho=500,T=650)", if97i_t3_h (rho, T), 1863430.19, 0.5E-2);
  approx ("u(rho=500,T=650)", if97i_t3_u (rho, T), 1812262.79, 0.5E-2);
  approx ("s(rho=500,T=650)", if97i_t3_s (rho, T), 4054.27273, 0.5E-5);
  approx ("cp(rho=500,T=650)", if97i_t3_cp (rho, T), 13893.5717, 0.5E-4);
  approx ("w(rho=500,T=650)", if97i_t3_w (rho, T), 502.005554, 0.5E-6);

  rho = 200.0;
  T = 650.0;

  if97i_t3 (&prop, rho, T);
  approx ("if_p(rho=200,T=650)", prop.if_p, 22293064.3, 0.5E-1);
  approx ("if_h(rho=200,T=650)", prop.if_h, 2375124.01, 0.5E-2);
  approx ("if_u(rho=200,T=650)", prop.if_u, 2263658.68, 0.5E-2);
  approx ("if_s(rho=200,T=650)", prop.if_s, 4854.38792, 0.5E-5);
  approx ("if_cp(rho=200,T=650)", prop.if_cp, 44657.9342, 0.5E-4);
  approx ("if_w(rho=200,T=650)", prop.if_w, 383.444594, 0.5E-6);

  approx ("p(rho=200,T=650)", if97i_t3_p (rho, T), 22293064.3, 0.5E-1);
  approx ("h(rho=200,T=650)", if97i_t3_h (rho, T), 2375124.01, 0.5E-2);
  approx ("u(rho=200,T=650)", if97i_t3_u (rho, T), 2263658.68, 0.5E-2);
  approx ("s(rho=200,T=650)", if97i_t3_s (rho, T), 4854.38792, 0.5E-5);
  approx ("cp(rho=200,T=650)", if97i_t3_cp (rho, T), 44657.9342, 0.5E-4);
  approx ("w(rho=200,T=650)", if97i_t3_w (rho, T), 383.444594, 0.5E-6);

  rho = 500.0;
  T = 750.0;

  if97i_t3 (&prop, rho, T);
  approx ("if_p(rho=500,T=750)", prop.if_p, 78309563.9, 0.5E-1);
  approx ("if_h(rho=500,T=750)", prop.if_h, 2258688.45, 0.5E-2);
  approx ("if_u(rho=500,T=750)", prop.if_u, 2102069.32, 0.5E-2);
  approx ("if_s(rho=500,T=750)", prop.if_s, 4469.71906, 0.5E-5);
  approx ("if_cp(rho=500,T=750)", prop.if_cp, 6341.65359, 0.5E-5);
  approx ("if_w(rho=500,T=750)", prop.if_w, 760.696041, 0.5E-6);

  approx ("p(rho=500,T=750)", if97i_t3_p (rho, T), 78309563.9, 0.5E-1);
  approx ("h(rho=500,T=750)", if97i_t3_h (rho, T), 2258688.45, 0.5E-2);
  approx ("u(rho=500,T=750)", if97i_t3_u (rho, T), 2102069.32, 0.5E-2);
  approx ("s(rho=500,T=750)", if97i_t3_s (rho, T), 4469.71906, 0.5E-5);
  approx ("cp(rho=500,T=750)", if97i_t3_cp (rho, T), 6341.65359, 0.5E-5);
  approx ("w(rho=500,T=750)", if97i_t3_w (rho, T), 760.696041, 0.5E-6);

  printf ("\nSaturation Line");
  printf ("\n~~~~~~~~~~~~~~~\n");

  /* Table 3.35.  */
  approx ("p_s(T_s=300)", if97i_p_s (300.0), 3536.58941, 0.5E-5);
  approx ("p_s(T_s=500)", if97i_p_s (500.0), 2638897.76, 0.5E-2);
  approx ("p_s(T_s=600)", if97i_p_s (600.0), 12344314.6, 0.5E-1);

  /* Table 3.36.  */
  approx ("T_s(p_s=0.1M)", if97i_t_s (0.1E+6), 372.755919, 0.5E-6);
  approx ("T_s(p_s=1M)", if97i_t_s (1.0E+6), 453.035632, 0.5E-6);
  approx ("T_s(p_s=10M)", if97i_t_s (10.0E+6), 584.149488, 0.5E-6);

  printf ("\nBoundary Line between Regions 2 and 3");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  p = 16529164.3;
  T = 623.15;

  approx ("p_b23(T_b23=623.15)", if97i_p_b23 (T), p, 0.5E-1);
  approx ("T_b23(p_b23=16529164.3)", if97i_t_b23 (p), T, 0.5E-2);

  printf ("\nBoundary Line between Subregions 2b and 2c");
  printf ("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");

  p = 100000000.0;
  h = 3516004.323;

  approx ("p_b2bc(h_b2bc=3516004.323)", if97i_p_b2bc (h), p, 0.5E-1);
  approx ("h_b2bc(p_b2bc=100000000.0)", if97i_h_b2bc (p), h, 0.5E-3);

  printf ("\nSummary");
  printf ("\n~~~~~~~\n");

  printf ("%d out of %d tests failed, ", errors, tests);
  printf ("%G%% of the tests succeed\n", 100.0 * (tests - errors) / tests);

  exit ((errors == 0) ? EXIT_SUCCESS : EXIT_FAILURE);
}
