/* if97t.c --- IAPWS-IF97 temperature interface.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-errno.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"


/* Calculate all properties at pressure P and temperature T.

   The result is returned in the variable PROP.  If PROP is a null
   pointer, then the storage space for PROP will be allocated using
   'if97_create'.

   Return value is the address of PROP, or a null pointer if an error
   occurs.  */
struct if97 *
if97t (struct if97 *prop, double p, double t)
{
  static struct if97 *(*map[6]) (struct if97 *, double, double) =
    {
      NULL,
      if97i_t1,
      if97i_t2,
      if97i_t3,
      NULL,
      if97i_t5,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, NULL);

  if (prop == NULL)
    {
      prop = if97_create (1);
      if (prop == NULL)
	return NULL;
    }

  return map[i] (prop, p, t);
}


/* Return the specific volume of water in m^3/kg at pressure P and
   temperature T.  */
double
if97t_v (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_v,
      if97i_t2_v,
      if97i_t3_v,
      NULL,
      if97i_t5_v,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the specific enthalpy of water in J/kg at pressure P and
   temperature T.  */
double if97t_h (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_h,
      if97i_t2_h,
      if97i_t3_h,
      NULL,
      if97i_t5_h,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the specific internal energy of water in J/kg at pressure P
   and temperature T.  */
double
if97t_u (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_u,
      if97i_t2_u,
      if97i_t3_u,
      NULL,
      if97i_t5_u,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the specific entropy of water in J/kg at pressure P and
   temperature T.  */
double if97t_s (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_s,
      if97i_t2_s,
      if97i_t3_s,
      NULL,
      if97i_t5_s,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the specific isobaric heat capacity of water in J/kg/K at
   pressure P and temperature T.  */
double if97t_cp (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_cp,
      if97i_t2_cp,
      if97i_t3_cp,
      NULL,
      if97i_t5_cp,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the specific isochoric heat capacity of water in J/kg/K at
   pressure P and temperature T.  */
double if97t_cv (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_cv,
      if97i_t2_cv,
      if97i_t3_cv,
      NULL,
      if97i_t5_cv,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the speed of sound of water in m/s at pressure P and
   temperature T.  */
double if97t_w (double p, double t)
{
  static double (*map[6]) (double, double) =
    {
      NULL,
      if97i_t1_w,
      if97i_t2_w,
      if97i_t3_w,
      NULL,
      if97i_t5_w,
    };

  int i;

  i = if97t_region (p, t);
  if (i == -1)
    set_errno_and_return (EDOM, 0.0);

  return map[i] (p, t);
}


/* Return the IAPWS-IF97 region corresponding to pressure P and
   temperature T.  Value is either 1, 2, 3, or 5.  The saturation
   line (region 4) is never considered by this function.

   The following precedence rules apply if the arguments are on the
   boundary of two regions:

        * 1-2 boundary: region 1
        * 1-3 boundary: region 1
        * 2-3 boundary: region 3
        * 2-5 boundary: region 2

   A value of -1 is returned if the pressure p or the temperature T
   is outside the argument range.  */
int
if97t_region (double p, double t)
{
  if (p < P0 || p > 100.0E+6)
    return -1;

  if (t < T0 || t > 2273.15)
    return -1;

  if (t <= 623.15)
    {
      double ps;

      ps = if97i_p_s (t);
      if (p < ps)
	return 2;

      return 1;
    }

  if (t <= 863.15)
    {
      double p23;

      p23 = if97i_p_b23 (t);
      if (p < p23)
	return 2;

      return 3;
    }

  if (t <= 1073.15)
    return 2;

  if (p > 10.0E+6)
    return -1;

  return 5;
}
