/* s1.c --- equations for region 1 (entropy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.7: Coefficients and exponents of equation (3.5).  */
struct t37
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t37 t37[] =
  {
    { 1, 0,  0,  0.17478268058307E+03},
    { 2, 0,  1,  0.34806930892873E+02},
    { 3, 0,  2,  0.65292584978455E+01},
    { 4, 0,  3,  0.33039981775489E+00},
    { 5, 0, 11, -0.19281382923196E-06},
    { 6, 0, 31, -0.24909197244573E-22},
    { 7, 1,  0, -0.26107636489332E+00},
    { 8, 1,  1,  0.22592965981586E+00},
    { 9, 1,  2, -0.64256463395226E-01},
    {10, 1,  3,  0.78876289270526E-02},
    {11, 1, 12,  0.35672110607366E-09},
    {12, 1, 31,  0.17332496994895E-23},
    {13, 2,  0,  0.56608900654837E-03},
    {14, 2,  1, -0.32635483139717E-03},
    {15, 2,  2,  0.44778286690632E-04},
    {16, 2,  9, -0.51322156908507E-09},
    {17, 2, 31, -0.42522657042207E-25},
    {18, 3, 10,  0.26400441360689E-12},
    {19, 3, 32,  0.78124600459723E-28},
    {20, 4, 32, -0.30732199903668E-30},
  };

#define I(i) t37[i].I
#define J(i) t37[i].J
#define n(i) t37[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference entropy.  */
#define s_star 1.0E+3

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_s1_p_star (void)
{
  return p_star;
}


/* Return the reference entropy.  */
double
if97i_s1_s_star (void)
{
  return s_star;
}


/* Return the reference temperature.  */
double
if97i_s1_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_s1_pi (double p)
{
  return p / p_star;
}


/* Return the reduced entropy.  */
double
if97i_s1_sigma (double s)
{
  return s / s_star;
}


/* Return the reduced temperature when using equation (3.5).  */
double
if97i_s1_theta (double pi, double sigma)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 20; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (sigma + 2.0), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature of water at pressure P and entropy S.  */
double
if97i_s1_t (double p, double s)
{
  double pi, sigma, theta;

  pi = if97i_s1_pi (p);
  sigma = if97i_s1_sigma (s);

  theta = if97i_s1_theta (pi, sigma);

  return theta * t_star;
}
