/* s2a.c --- equations for subregion 2a (entropy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.25: Coefficients and exponents of equation (3.18).  */
struct t325
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    double I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t325 t325[] =
  {
    { 1, -1.5 , -24.0, -0.39235983861984E+06},
    { 2, -1.5 , -23.0,  0.51526573827270E+06},
    { 3, -1.5 , -19.0,  0.40482443161048E+05},
    { 4, -1.5 , -13.0, -0.32193790923902E+03},
    { 5, -1.5 , -11.0,  0.96961424218694E+02},
    { 6, -1.5 , -10.0, -0.22867846371773E+02},
    { 7, -1.25, -19.0, -0.44942914124357E+06},
    { 8, -1.25, -15.0, -0.50118336020166E+04},
    { 9, -1.25,  -6.0,  0.35684463560015E+00},
    {10, -1.0 , -26.0,  0.44235335848190E+05},
    {11, -1.0 , -21.0, -0.13673388811708E+05},
    {12, -1.0 , -17.0,  0.42163260207864E+06},
    {13, -1.0 , -16.0,  0.22516925837475E+05},
    {14, -1.0 ,  -9.0,  0.47442144865646E+03},
    {15, -1.0 ,  -8.0, -0.14931130797647E+03},
    {16, -0.75, -15.0, -0.19781126320452E+06},
    {17, -0.75, -14.0, -0.23554399470760E+05},
    {18, -0.5 , -26.0, -0.19070616302076E+05},
    {19, -0.5 , -13.0,  0.55375669883164E+05},
    {20, -0.5 ,  -9.0,  0.38293691437363E+04},
    {21, -0.5 ,  -7.0, -0.60391860580567E+03},
    {22, -0.25, -27.0,  0.19363102620331E+04},
    {23, -0.25, -25.0,  0.42660643698610E+04},
    {24, -0.25, -11.0, -0.59780638872718E+04},
    {25, -0.25,  -6.0, -0.70401463926862E+03},
    {26,  0.25,   1.0,  0.33836784107553E+03},
    {27,  0.25,   4.0,  0.20862786635187E+02},
    {28,  0.25,   8.0,  0.33834172656196E-01},
    {29,  0.25,  11.0, -0.43124428414893E-04},
    {30,  0.5 ,   0.0,  0.16653791356412E+03},
    {31,  0.5 ,   1.0, -0.13986292055898E+03},
    {32,  0.5 ,   5.0, -0.78849547999872E+00},
    {33,  0.5 ,   6.0,  0.72132411753872E-01},
    {34,  0.5 ,  10.0, -0.59754839398283E-02},
    {35,  0.5 ,  14.0, -0.12141358953904E-04},
    {36,  0.5 ,  16.0,  0.23227096733871E-06},
    {37,  0.75,   0.0, -0.10538463566194E+02},
    {38,  0.75,   4.0,  0.20718925496502E+01},
    {39,  0.75,   9.0, -0.72193155260427E-01},
    {40,  0.75,  17.0,  0.20749887081120E-06},
    {41,  1.0 ,   7.0, -0.18340657911379E-01},
    {42,  1.0 ,  18.0,  0.29036272348696E-06},
    {43,  1.25,   3.0,  0.21037527893619E+00},
    {44,  1.25,  15.0,  0.25681239729999E-03},
    {45,  1.5 ,   5.0, -0.12799002933781E-01},
    {46,  1.5 ,  18.0, -0.82198102652018E-05},
  };

#define I(i) t325[i].I
#define J(i) t325[i].J
#define n(i) t325[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference entropy.  */
#define s_star 2.0E+3

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_s2a_p_star (void)
{
  return p_star;
}


/* Return the reference entropy.  */
double
if97i_s2a_s_star (void)
{
  return s_star;
}


/* Return the reference temperature.  */
double
if97i_s2a_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_s2a_pi (double p)
{
  return p / p_star;
}


/* Return the reduced entropy.  */
double
if97i_s2a_sigma (double s)
{
  return s / s_star;
}


/* Return the reduced temperature.  */
double
if97i_s2a_theta (double pi, double sigma)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 46; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (sigma - 2.0), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and entropy S.  */
double
if97i_s2a_t (double p, double s)
{
  double pi, sigma, theta;

  pi = if97i_s2a_pi (p);
  sigma = if97i_s2a_sigma (s);

  theta = if97i_s2a_theta (pi, sigma);

  return theta * t_star;
}
