/* s2b.c --- equations for subregion 2b (entropy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.26: Coefficients and exponents of equation (3.19).  */
struct t326
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    double I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t326 t326[] =
  {
    { 1, -6.0,  0.0,  0.31687665083497E+06},
    { 2, -6.0, 11.0,  0.20864175881858E+02},
    { 3, -5.0,  0.0, -0.39859399803599E+06},
    { 4, -5.0, 11.0, -0.21816058518877E+02},
    { 5, -4.0,  0.0,  0.22369785194242E+06},
    { 6, -4.0,  1.0, -0.27841703445817E+04},
    { 7, -4.0, 11.0,  0.99207436071480E+01},
    { 8, -3.0,  0.0, -0.75197512299157E+05},
    { 9, -3.0,  1.0,  0.29708605951158E+04},
    {10, -3.0, 11.0, -0.34406878548526E+01},
    {11, -3.0, 12.0,  0.38815564249115E+00},
    {12, -2.0,  0.0,  0.17511295085750E+05},
    {13, -2.0,  1.0, -0.14237112854449E+04},
    {14, -2.0,  6.0,  0.10943803364167E+01},
    {15, -2.0, 10.0,  0.89971619308495E+00},
    {16, -1.0,  0.0, -0.33759740098958E+04},
    {17, -1.0,  1.0,  0.47162885818355E+03},
    {18, -1.0,  5.0, -0.19188241993679E+01},
    {19, -1.0,  8.0,  0.41078580492196E+00},
    {20, -1.0,  9.0, -0.33465378172097E+00},
    {21,  0.0,  0.0,  0.13870034777505E+04},
    {22,  0.0,  1.0, -0.40663326195838E+03},
    {23,  0.0,  2.0,  0.41727347159610E+02},
    {24,  0.0,  4.0,  0.21932549434532E+01},
    {25,  0.0,  5.0, -0.10320050009077E+01},
    {26,  0.0,  6.0,  0.35882943516703E+00},
    {27,  0.0,  9.0,  0.52511453726066E-02},
    {28,  1.0,  0.0,  0.12838916450705E+02},
    {29,  1.0,  1.0, -0.28642437219381E+01},
    {30,  1.0,  2.0,  0.56912683664855E+00},
    {31,  1.0,  3.0, -0.99962954584931E-01},
    {32,  1.0,  7.0, -0.32632037778459E-02},
    {33,  1.0,  8.0,  0.23320922576723E-03},
    {34,  2.0,  0.0, -0.15334809857450E+00},
    {35,  2.0,  1.0,  0.29072288239902E-01},
    {36,  2.0,  5.0,  0.37534702741167E-03},
    {37,  3.0,  0.0,  0.17296691702411E-02},
    {38,  3.0,  1.0, -0.38556050844504E-03},
    {39,  3.0,  3.0, -0.35017712292608E-04},
    {40,  4.0,  0.0, -0.14566393631492E-04},
    {41,  4.0,  1.0,  0.56420857267269E-05},
    {42,  5.0,  0.0,  0.41286150074605E-07},
    {43,  5.0,  1.0, -0.20684671118824E-07},
    {44,  5.0,  2.0,  0.16409393674725E-08},
  };

#define I(i) t326[i].I
#define J(i) t326[i].J
#define n(i) t326[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference entropy.  */
#define s_star 0.7853E+3

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_s2b_p_star (void)
{
  return p_star;
}


/* Return the reference entropy.  */
double
if97i_s2b_s_star (void)
{
  return s_star;
}


/* Return the reference temperature.  */
double
if97i_s2b_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_s2b_pi (double p)
{
  return p / p_star;
}


/* Return the reduced entropy.  */
double
if97i_s2b_sigma (double s)
{
  return s / s_star;
}


/* Return the reduced temperature.  */
double
if97i_s2b_theta (double pi, double sigma)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 44; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (10.0 - sigma), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and entropy S.  */
double
if97i_s2b_t (double p, double s)
{
  double pi, sigma, theta;

  pi = if97i_s2b_pi (p);
  sigma = if97i_s2b_sigma (s);

  theta = if97i_s2b_theta (pi, sigma);

  return theta * t_star;
}
