/* s2c.c --- equations for subregion 2c (entropy interface).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.27: Coefficients and exponents of equation (3.20).  */
struct t327
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    double I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t327 t327[] =
  {
    { 1, -2.0, 0.0,  0.90968501005365E+03},
    { 2, -2.0, 1.0,  0.24045667088420E+04},
    { 3, -1.0, 0.0, -0.59162326387130E+03},
    { 4,  0.0, 0.0,  0.54145404128074E+03},
    { 5,  0.0, 1.0, -0.27098308411192E+03},
    { 6,  0.0, 2.0,  0.97976525097926E+03},
    { 7,  0.0, 3.0, -0.46966772959435E+03},
    { 8,  1.0, 0.0,  0.14399274604723E+02},
    { 9,  1.0, 1.0, -0.19104204230429E+02},
    {10,  1.0, 3.0,  0.53299167111971E+01},
    {11,  1.0, 4.0, -0.21252975375934E+02},
    {12,  2.0, 0.0, -0.31147334413760E+00},
    {13,  2.0, 1.0,  0.60334840894623E+00},
    {14,  2.0, 2.0, -0.42764839702509E-01},
    {15,  3.0, 0.0,  0.58185597255259E-02},
    {16,  3.0, 1.0, -0.14597008284753E-01},
    {17,  3.0, 5.0,  0.56631175631027E-02},
    {18,  4.0, 0.0, -0.76155864584577E-04},
    {19,  4.0, 1.0,  0.22440342919332E-03},
    {20,  4.0, 4.0, -0.12561095013413E-04},
    {21,  5.0, 0.0,  0.63323132660934E-06},
    {22,  5.0, 1.0, -0.20541989675375E-05},
    {23,  5.0, 2.0,  0.36405370390082E-07},
    {24,  6.0, 0.0, -0.29759897789215E-08},
    {25,  6.0, 1.0,  0.10136618529763E-07},
    {26,  7.0, 0.0,  0.59925719692351E-11},
    {27,  7.0, 1.0, -0.20677870105164E-10},
    {28,  7.0, 3.0, -0.20874278181886E-10},
    {29,  7.0, 4.0,  0.10162166825089E-09},
    {30,  7.0, 5.0, -0.16429828281347E-09},
  };

#define I(i) t327[i].I
#define J(i) t327[i].J
#define n(i) t327[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference entropy.  */
#define s_star 2.9251E+3

/* Reference temperature.  */
#define t_star 1.0


/* Return the reference pressure.  */
double
if97i_s2c_p_star (void)
{
  return p_star;
}


/* Return the reference entropy.  */
double
if97i_s2c_s_star (void)
{
  return s_star;
}


/* Return the reference temperature.  */
double
if97i_s2c_t_star (void)
{
  return t_star;
}



/* Return the reduced pressure.  */
double
if97i_s2c_pi (double p)
{
  return p / p_star;
}


/* Return the reduced entropy.  */
double
if97i_s2c_sigma (double s)
{
  return s / s_star;
}


/* Return the reduced temperature.  */
double
if97i_s2c_theta (double pi, double sigma)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 30; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (2.0 - sigma), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the temperature at pressure P and entropy S.  */
double
if97i_s2c_t (double p, double s)
{
  double pi, sigma, theta;

  pi = if97i_s2c_pi (p);
  sigma = if97i_s2c_sigma (s);

  theta = if97i_s2c_theta (pi, sigma);

  return theta * t_star;
}
