/* t1.c --- basic equation for region 1.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.1: Coefficients and exponents of equation (3.1).  */
struct t31
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t31 t31[] =
  {
    { 1,  0,  -2,  0.14632971213167E+00},
    { 2,  0,  -1, -0.84548187169114E+00},
    { 3,  0,   0, -0.37563603672040E+01},
    { 4,  0,   1,  0.33855169168385E+01},
    { 5,  0,   2, -0.95791963387872E+00},
    { 6,  0,   3,  0.15772038513228E+00},
    { 7,  0,   4, -0.16616417199501E-01},
    { 8,  0,   5,  0.81214629983568E-03},
    { 9,  1,  -9,  0.28319080123804E-03},
    {10,  1,  -7, -0.60706301565874E-03},
    {11,  1,  -1, -0.18990068218419E-01},
    {12,  1,   0, -0.32529748770505E-01},
    {13,  1,   1, -0.21841717175414E-01},
    {14,  1,   3, -0.52838357969930E-04},
    {15,  2,  -3, -0.47184321073267E-03},
    {16,  2,   0, -0.30001780793026E-03},
    {17,  2,   1,  0.47661393906987E-04},
    {18,  2,   3, -0.44141845330846E-05},
    {19,  2,  17, -0.72694996297594E-15},
    {20,  3,  -4, -0.31679644845054E-04},
    {21,  3,   0, -0.28270797985312E-05},
    {22,  3,   6, -0.85205128120103E-09},
    {23,  4,  -5, -0.22425281908000E-05},
    {24,  4,  -2, -0.65171222895601E-06},
    {25,  4,  10, -0.14341729937924E-12},
    {26,  5,  -8, -0.40516996860117E-06},
    {27,  8, -11, -0.12734301741641E-08},
    {28,  8,  -6, -0.17424871230634E-09},
    {29, 21, -29, -0.68762131295531E-18},
    {30, 23, -31,  0.14478307828521E-19},
    {31, 29, -38,  0.26335781662795E-22},
    {32, 30, -39, -0.11947622640071E-22},
    {33, 31, -40,  0.18228094581404E-23},
    {34, 32, -41, -0.93537087292458E-25},
  };

#define I(i) t31[i].I
#define J(i) t31[i].J
#define n(i) t31[i].n

/* Reference pressure.  */
#define p_star 16.53E+6

/* Reference temperature.  */
#define t_star 1386.0


/* Return the reference pressure.  */
double
if97i_t1_p_star (void)
{
  return p_star;
}


/* Return the reference temperature.  */
double
if97i_t1_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_t1_pi (double p)
{
  return p / p_star;
}


/* Return the inverse reduced temperature.  */
double
if97i_t1_tau (double t)
{
  return t_star / t;
}


/* Return the dimensionless Gibbs free energy.  */
double
if97i_t1_gamma (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, (7.1 - pi), I(i));
      prod (tem, (tau - 1.222), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the first partial derivative of the dimensionless Gibbs
   free energy to the reduced pressure at constant inverse reduced
   temperature.  */
double
if97i_t1_gamma_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, (7.1 - pi), (I(i) - 1));
      prod (tem, (tau - 1.222), J(i));

      sum -= tem;
    }

  return sum;
}


/* Return the first partial derivative of the dimensionless Gibbs
   free energy to the inverse reduced temperature at constant reduced
   pressure.  */
double
if97i_t1_gamma_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, (7.1 - pi), I(i));
      prod (tem, J(i), 1);
      prod (tem, (tau - 1.222), (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the dimensionless Gibbs
   free energy to the reduced pressure at constant inverse reduced
   temperature.  */
double
if97i_t1_gamma_pi_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, I(i) * (I(i) - 1), 1);
      prod (tem, (7.1 - pi), (I(i) - 2));
      prod (tem, (tau - 1.222), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the dimensionless Gibbs
   free energy to the inverse reduced temperature at constant reduced
   pressure.  */
double
if97i_t1_gamma_tau_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, (7.1 - pi), I(i));
      prod (tem, J(i) * (J(i) - 1), 1);
      prod (tem, (tau - 1.222), (J(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the dimensionless Gibbs free
   energy to the reduced pressure and the inverse reduced temperature.  */
double
if97i_t1_gamma_pi_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 34; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, (7.1 - pi), (I(i) - 1));
      prod (tem, J(i), 1);
      prod (tem, (tau - 1.222), (J(i) - 1));

      sum -= tem;
    }

  return sum;
}


/* Calculate all properties at pressure P and temperature T.
   The result is returned in the variable PROP.

   Return value is the address of PROP.  */
struct if97 *
if97i_t1 (struct if97 *prop, double p, double t)
{
  double pi, tau;
  double gamma, gamma_pi, gamma_tau;
  double gamma_pi_pi, gamma_tau_tau, gamma_pi_tau;
  double num, v, h, s, u, cp, cv, w;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma = if97i_t1_gamma (pi, tau);
  gamma_pi = if97i_t1_gamma_pi (pi, tau);
  gamma_tau = if97i_t1_gamma_tau (pi, tau);
  gamma_pi_pi = if97i_t1_gamma_pi_pi (pi, tau);
  gamma_tau_tau = if97i_t1_gamma_tau_tau (pi, tau);
  gamma_pi_tau = if97i_t1_gamma_pi_tau (pi, tau);

  num = sq (gamma_pi - tau * gamma_pi_tau);

  v = pi * gamma_pi;
  h = tau * gamma_tau;
  u = h - v;
  s = h - gamma;
  cp = sq (tau) * gamma_tau_tau;  /* Wrong sign!  */
  cv = num / gamma_pi_pi - cp;
  w = sq (gamma_pi) / (num / cp - gamma_pi_pi);

  prop->if_p = p;
  prop->if_t = t;
  prop->if_v = R * t / p * v;
  prop->if_h = R * t * h;
  prop->if_u = R * t * u;
  prop->if_s = R * s;
  prop->if_cp = - R * cp;  /* Correct sign!  */
  prop->if_cv = R * cv;
  prop->if_w = sqrt (R * t * w);

  return prop;
}


/* Return the specific volume at pressure P and temperature T.  */
double
if97i_t1_v (double p, double t)
{
  double pi, tau;
  double gamma_pi;
  double v;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_pi = if97i_t1_gamma_pi (pi, tau);

  v = pi * gamma_pi;

  return R * t / p * v;
}


/* Return the specific enthalpy at pressure P and temperature T.  */
double
if97i_t1_h (double p, double t)
{
  double pi, tau;
  double gamma_tau;
  double h;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_tau = if97i_t1_gamma_tau (pi, tau);

  h = tau * gamma_tau;

  return R * t * h;
}


/* Return the specific internal energy at pressure P and temperature T.  */
double
if97i_t1_u (double p, double t)
{
  double pi, tau;
  double gamma_pi, gamma_tau;
  double v, h, u;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_pi = if97i_t1_gamma_pi (pi, tau);
  gamma_tau = if97i_t1_gamma_tau (pi, tau);

  v = pi * gamma_pi;
  h = tau * gamma_tau;
  u = h - v;

  return R * t * u;
}


/* Return the specific entropy at pressure P and temperature T.  */
double
if97i_t1_s (double p, double t)
{
  double pi, tau;
  double gamma, gamma_tau;
  double h, s;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma = if97i_t1_gamma (pi, tau);
  gamma_tau = if97i_t1_gamma_tau (pi, tau);

  h = tau * gamma_tau;
  s = h - gamma;

  return R * s;
}


/* Return the specific isobaric heat capacity at pressure P and
   temperature T.  */
double
if97i_t1_cp (double p, double t)
{
  double pi, tau;
  double gamma_tau_tau;
  double cp;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_tau_tau = if97i_t1_gamma_tau_tau (pi, tau);

  cp = tau * tau * gamma_tau_tau;

  return - R * cp;
}


/* Return the specific isochoric heat capacity at pressure P and
   temperature T.  */
double
if97i_t1_cv (double p, double t)
{
  double pi, tau;
  double gamma_pi;
  double gamma_pi_pi, gamma_tau_tau, gamma_pi_tau;
  double num, cp, cv;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_pi = if97i_t1_gamma_pi (pi, tau);
  gamma_pi_pi = if97i_t1_gamma_pi_pi (pi, tau);
  gamma_tau_tau = if97i_t1_gamma_tau_tau (pi, tau);
  gamma_pi_tau = if97i_t1_gamma_pi_tau (pi, tau);

  num = sq (gamma_pi - tau * gamma_pi_tau);

  cp = tau * tau * gamma_tau_tau;
  cv = num / gamma_pi_pi - cp;

  return R * cv;
}


/* Return the speed of sound at pressure P and temperature T.  */
double
if97i_t1_w (double p, double t)
{
  double pi, tau;
  double gamma_pi;
  double gamma_pi_pi, gamma_tau_tau, gamma_pi_tau;
  double num, cp, w;

  pi = if97i_t1_pi (p);
  tau = if97i_t1_tau (t);

  gamma_pi = if97i_t1_gamma_pi (pi, tau);
  gamma_pi_pi = if97i_t1_gamma_pi_pi (pi, tau);
  gamma_tau_tau = if97i_t1_gamma_tau_tau (pi, tau);
  gamma_pi_tau = if97i_t1_gamma_pi_tau (pi, tau);

  num = sq (gamma_pi - tau * gamma_pi_tau);

  cp = tau * tau * gamma_tau_tau;
  w = sq (gamma_pi) / (num / cp - gamma_pi_pi);

  return sqrt (R * t * w);
}
