/* t2.c --- basic equation for region 2.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.10: Coefficients and exponents of equation (3.9).  */
struct t310
  {
    /* Index.  */
    int i;

    /* Exponent.  */
    int J;

    /* Coefficient.  */
    double n;
  };

static const struct t310 t310[] =
  {
    {1,  0, -0.96927686500217E+01},
    {2,  1,  0.10086655968018E+02},
    {3, -5, -0.56087911283020E-02},
    {4, -4,  0.71452738081455E-01},
    {5, -3, -0.40710498223928E+00},
    {6, -2,  0.14240819171444E+01},
    {7, -1, -0.43839511319450E+01},
    {8,  2, -0.28408632460772E+00},
    {9,  3,  0.21268463753307E-01},
  };

#define J0(i) t310[i].J
#define n0(i) t310[i].n

/* Table 3.11: Coefficients and exponents of equation (3.10).  */
struct t311
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    int I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t311 t311[] =
  {
    { 1,  1,  0, -0.17731742473213E-02},
    { 2,  1,  1, -0.17834862292358E-01},
    { 3,  1,  2, -0.45996013696365E-01},
    { 4,  1,  3, -0.57581259083432E-01},
    { 5,  1,  6, -0.50325278727930E-01},
    { 6,  2,  1, -0.33032641670203E-04},
    { 7,  2,  2, -0.18948987516315E-03},
    { 8,  2,  4, -0.39392777243355E-02},
    { 9,  2,  7, -0.43797295650573E-01},
    {10,  2, 36, -0.26674547914087E-04},
    {11,  3,  0,  0.20481737692309E-07},
    {12,  3,  1,  0.43870667284435E-06},
    {13,  3,  3, -0.32277677238570E-04},
    {14,  3,  6, -0.15033924542148E-02},
    {15,  3, 35, -0.40668253562649E-01},
    {16,  4,  1, -0.78847309559367E-09},
    {17,  4,  2,  0.12790717852285E-07},
    {18,  4,  3,  0.48225372718507E-06},
    {19,  5,  7,  0.22922076337661E-05},
    {20,  6,  3, -0.16714766451061E-10},
    {21,  6, 16, -0.21171472321355E-02},
    {22,  6, 35, -0.23895741934104E+02},
    {23,  7,  0, -0.59059564324270E-17},
    {24,  7, 11, -0.12621808899101E-05},
    {25,  7, 25, -0.38946842435739E-01},
    {26,  8,  8,  0.11256211360459E-10},
    {27,  8, 36, -0.82311340897998E+01},
    {28,  9, 13,  0.19809712802088E-07},
    {29, 10,  4,  0.10406965210174E-18},
    {30, 10, 10, -0.10234747095929E-12},
    {31, 10, 14, -0.10018179379511E-08},
    {32, 16, 29, -0.80882908646985E-10},
    {33, 16, 50,  0.10693031879409E+00},
    {34, 18, 57, -0.33662250574171E+00},
    {35, 20, 20,  0.89185845355421E-24},
    {36, 20, 35,  0.30629316876232E-12},
    {37, 20, 48, -0.42002467698208E-05},
    {38, 21, 21, -0.59056029685639E-25},
    {39, 22, 53,  0.37826947613457E-05},
    {40, 23, 39, -0.12768608934681E-14},
    {41, 24, 26,  0.73087610595061E-28},
    {42, 24, 40,  0.55414715350778E-16},
    {43, 24, 58, -0.94369707241210E-06},
  };

#define I(i) t311[i].I
#define J(i) t311[i].J
#define n(i) t311[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference temperature.  */
#define t_star 540.0


/* Return the reference pressure.  */
double
if97i_t2_p_star (void)
{
  return p_star;
}


/* Return the reference temperature.  */
double
if97i_t2_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_t2_pi (double p)
{
  return p / p_star;
}


/* Return the inverse reduced temperature.  */
double
if97i_t2_tau (double t)
{
  return t_star / t;
}


/* Return the ideal-gas part of the dimensionless Gibbs free energy.  */
double
if97i_t2_gamma_0 (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 9; ++i)
    {
      double tem = n0(i);

      prod (tem, tau, J0(i));

      sum += tem;
    }

  return log (pi) + sum;
}


/* Return the first partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t2_gamma_0_pi (double pi, double tau __attribute__ ((unused)))
{
  return 1.0 / pi;
}


/* Return the first partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t2_gamma_0_tau (double pi __attribute__ ((unused)), double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 9; ++i)
    {
      double tem = n0(i);

      prod (tem, J0(i), 1);
      prod (tem, tau, (J0(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t2_gamma_0_pi_pi (double pi, double tau __attribute__ ((unused)))
{
  return -1.0 / (pi * pi);
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t2_gamma_0_tau_tau (double pi __attribute__ ((unused)), double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 9; ++i)
    {
      double tem = n0(i);

      prod (tem, J0(i) * (J0(i) - 1), 1);
      prod (tem, tau, (J0(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure and the
   inverse reduced temperature.  */
double
if97i_t2_gamma_0_pi_tau (double pi __attribute__ ((unused)), double tau __attribute__ ((unused)))
{
  return 0.0;
}


/* Return the residual part of the dimensionless Gibbs free energy.  */
double
if97i_t2_gamma_r (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, (tau - 0.5), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the first partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t2_gamma_r_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, pi, (I(i) - 1));
      prod (tem, (tau - 0.5), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the first partial derivative of the residual part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t2_gamma_r_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, J(i), 1);
      prod (tem, (tau - 0.5), (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t2_gamma_r_pi_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, I(i) * (I(i) - 1), 1);
      prod (tem, pi, (I(i) - 2));
      prod (tem, (tau - 0.5), J(i));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t2_gamma_r_tau_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, J(i) * (J(i) - 1), 1);
      prod (tem, (tau - 0.5), (J(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure and the
   inverse reduced temperature.  */
double
if97i_t2_gamma_r_pi_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 43; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, pi, (I(i) - 1));
      prod (tem, J(i), 1);
      prod (tem, (tau - 0.5), (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Calculate all properties at pressure P and temperature T.
   The result is returned in the variable PROP.

   Return value is the address of PROP.  */
struct if97 *
if97i_t2 (struct if97 *prop, double p, double t)
{
  double pi, tau;
  double gamma_0, gamma_0_pi, gamma_0_tau;
  double gamma_0_pi_pi, gamma_0_tau_tau, gamma_0_pi_tau;
  double gamma_r, gamma_r_pi, gamma_r_tau;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, v, h, u, s, cp, cv, w;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0 = if97i_t2_gamma_0 (pi, tau);
  gamma_0_pi = if97i_t2_gamma_0_pi (pi, tau);
  gamma_0_tau = if97i_t2_gamma_0_tau (pi, tau);
  gamma_0_pi_pi = if97i_t2_gamma_0_pi_pi (pi, tau);
  gamma_0_tau_tau = if97i_t2_gamma_0_tau_tau (pi, tau);
  gamma_0_pi_tau = if97i_t2_gamma_0_pi_tau (pi, tau);

  gamma_r = if97i_t2_gamma_r (pi, tau);
  gamma_r_pi = if97i_t2_gamma_r_pi (pi, tau);
  gamma_r_tau = if97i_t2_gamma_r_tau (pi, tau);
  gamma_r_pi_pi = if97i_t2_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t2_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t2_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * (gamma_r_pi - tau * gamma_r_pi_tau));
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  v = pi * (gamma_0_pi + gamma_r_pi);
  h = tau * (gamma_0_tau + gamma_r_tau);
  u = h - v;
  s = h - (gamma_0 + gamma_r);
  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  cv = cp - num / den;
  w = (1.0 + pi * gamma_r_pi * (2.0 + pi * gamma_r_pi)) / (den - num / cp);

  prop->if_p = p;
  prop->if_t = t;
  prop->if_v = R * t / p * v;
  prop->if_h = R * t * h;
  prop->if_u = R * t * u;
  prop->if_s = R * s;
  prop->if_cp = R * cp;
  prop->if_cv = R * cv;
  prop->if_w = sqrt (R * t * w);

  return prop;
}


/* Return the specific volume at pressure P and temperature T.  */
double
if97i_t2_v (double p, double t)
{
  double pi, tau;
  double gamma_0_pi;
  double gamma_r_pi;
  double v;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_pi = if97i_t2_gamma_0_pi (pi, tau);
  gamma_r_pi = if97i_t2_gamma_r_pi (pi, tau);

  v = pi * (gamma_0_pi + gamma_r_pi);

  return R * t / p * v;
}


/* Return the specific enthalpy at pressure P and temperature T.  */
double
if97i_t2_h (double p, double t)
{
  double pi, tau;
  double gamma_0_tau;
  double gamma_r_tau;
  double h;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_tau = if97i_t2_gamma_0_tau (pi, tau);
  gamma_r_tau = if97i_t2_gamma_r_tau (pi, tau);

  h = tau * (gamma_0_tau + gamma_r_tau);

  return R * t * h;
}


/* Return the specific internal energy at pressure P and temperature T.  */
double
if97i_t2_u (double p, double t)
{
  double pi, tau;
  double gamma_0_pi, gamma_0_tau;
  double gamma_r_pi, gamma_r_tau;
  double v, h, u;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_pi = if97i_t2_gamma_0_pi (pi, tau);
  gamma_0_tau = if97i_t2_gamma_0_tau (pi, tau);
  gamma_r_pi = if97i_t2_gamma_r_pi (pi, tau);
  gamma_r_tau = if97i_t2_gamma_r_tau (pi, tau);

  v = pi * (gamma_0_pi + gamma_r_pi);
  h = tau * (gamma_0_tau + gamma_r_tau);
  u = h - v;

  return R * t * u;
}


/* Return the specific entropy at pressure P and temperature T.  */
double
if97i_t2_s (double p, double t)
{
  double pi, tau;
  double gamma_0, gamma_0_tau;
  double gamma_r, gamma_r_tau;
  double h, s;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0 = if97i_t2_gamma_0 (pi, tau);
  gamma_0_tau = if97i_t2_gamma_0_tau (pi, tau);
  gamma_r = if97i_t2_gamma_r (pi, tau);
  gamma_r_tau = if97i_t2_gamma_r_tau (pi, tau);

  h = tau * (gamma_0_tau + gamma_r_tau);
  s = h - (gamma_0 + gamma_r);

  return R * s;
}


/* Return the specific isobaric heat capacity at pressure P and
   temperature T.  */
double
if97i_t2_cp (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_tau_tau;
  double cp;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_tau_tau = if97i_t2_gamma_0_tau_tau (pi, tau);
  gamma_r_tau_tau = if97i_t2_gamma_r_tau_tau (pi, tau);

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);

  return R * cp;
}


/* Return the specific isochoric heat capacity at pressure P and
   temperature T.  */
double
if97i_t2_cv (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_pi;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, cp, cv;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_tau_tau = if97i_t2_gamma_0_tau_tau (pi, tau);
  gamma_r_pi = if97i_t2_gamma_r_pi (pi, tau);
  gamma_r_pi_pi = if97i_t2_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t2_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t2_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * (gamma_r_pi - tau * gamma_r_pi_tau));
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  cv = cp - num / den;

  return R * cv;
}


/* Return the speed of sound at pressure P and temperature T.  */
double
if97i_t2_w (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_pi;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, cp, w;

  pi = if97i_t2_pi (p);
  tau = if97i_t2_tau (t);

  gamma_0_tau_tau = if97i_t2_gamma_0_tau_tau (pi, tau);
  gamma_r_pi = if97i_t2_gamma_r_pi (pi, tau);
  gamma_r_pi_pi = if97i_t2_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t2_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t2_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * (gamma_r_pi - tau * gamma_r_pi_tau));
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  w = (1.0 + pi * gamma_r_pi * (2.0 + pi * gamma_r_pi)) / (den - num / cp);

  return sqrt (R * t * w);
}
