/* t3.c --- basic equation for region 3.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.30: Coefficients and exponents of equation (3.21).  */
struct t330
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    double I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t330 t330[] =
  {
    { 1,  0.0,  0.0,  0.10658070028513E+01},
    { 2,  0.0,  0.0, -0.15732845290239E+02},
    { 3,  0.0,  1.0,  0.20944396974307E+02},
    { 4,  0.0,  2.0, -0.76867707878716E+01},
    { 5,  0.0,  7.0,  0.26185947787954E+01},
    { 6,  0.0, 10.0, -0.28080781148620E+01},
    { 7,  0.0, 12.0,  0.12053369696517E+01},
    { 8,  0.0, 23.0, -0.84566812812502E-02},
    { 9,  1.0,  2.0, -0.12654315477714E+01},
    {10,  1.0,  6.0, -0.11524407806681E+01},
    {11,  1.0, 15.0,  0.88521043984318E+00},
    {12,  1.0, 17.0, -0.64207765181607E+00},
    {13,  2.0,  0.0,  0.38493460186671E+00},
    {14,  2.0,  2.0, -0.85214708824206E+00},
    {15,  2.0,  6.0,  0.48972281541877E+01},
    {16,  2.0,  7.0, -0.30502617256965E+01},
    {17,  2.0, 22.0,  0.39420536879154E-01},
    {18,  2.0, 26.0,  0.12558408424308E+00},
    {19,  3.0,  0.0, -0.27999329698710E+00},
    {20,  3.0,  2.0,  0.13899799569460E+01},
    {21,  3.0,  4.0, -0.20189915023570E+01},
    {22,  3.0, 16.0, -0.82147637173963E-02},
    {23,  3.0, 26.0, -0.47596035734923E+00},
    {24,  4.0,  0.0,  0.43984074473500E-01},
    {25,  4.0,  2.0, -0.44476435428739E+00},
    {26,  4.0,  4.0,  0.90572070719733E+00},
    {27,  4.0, 26.0,  0.70522450087967E+00},
    {28,  5.0,  1.0,  0.10770512626332E+00},
    {29,  5.0,  3.0, -0.32913623258954E+00},
    {30,  5.0, 26.0, -0.50871062041158E+00},
    {31,  6.0,  0.0, -0.22175400873096E-01},
    {32,  6.0,  2.0,  0.94260751665092E-01},
    {33,  6.0, 26.0,  0.16436278447961E+00},
    {34,  7.0,  2.0, -0.13503372241348E-01},
    {35,  8.0, 26.0, -0.14834345352472E-01},
    {36,  9.0,  2.0,  0.57922953628084E-03},
    {37,  9.0, 26.0,  0.32308904703711E-02},
    {38, 10.0,  0.0,  0.80964802996215E-04},
    {39, 10.0,  1.0, -0.16557679795037E-03},
    {40, 11.0, 26.0, -0.44923899061815E-04},
  };

#define I(i) t330[i].I
#define J(i) t330[i].J
#define n(i) t330[i].n

/* Reference density.  */
#define rho_star RHOC

/* Reference temperature.  */
#define t_star TC


/* Return the reference density.  */
double
if97i_t3_rho_star (void)
{
  return rho_star;
}


/* Return the reference temperature.  */
double
if97i_t3_t_star (void)
{
  return t_star;
}


/* Return the reduced density.  */
double
if97i_t3_delta (double rho)
{
  return rho / rho_star;
}


/* Return the inverse reduced temperature.  */
double
if97i_t3_tau (double t)
{
  return t_star / t;
}


/* Return the dimensionless Helmholtz free energy.  */
double
if97i_t3_phi (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, delta, I(i));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return n(0) * log (delta) + sum;
}


/* Return the first partial derivative of the dimensionless Helmholtz
   free energy to the reduced density at constant inverse reduced
   temperature.  */
double
if97i_t3_phi_delta (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, delta, (I(i) - 1));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return n(0) / delta + sum;
}


/* Return the first partial derivative of the dimensionless Helmholtz
   free energy to the inverse reduced temperature at constant reduced
   density.  */
double
if97i_t3_phi_tau (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, delta, I(i));
      prod (tem, J(i), 1);
      prod (tem, tau, (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the dimensionless Helmholtz
   free energy to the reduced density at constant inverse reduced
   temperature.  */
double
if97i_t3_phi_delta_delta (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, I(i) * (I(i) - 1), 1);
      prod (tem, delta, (I(i) - 2));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return sum - n(0) / (sq (delta));
}


/* Return the second partial derivative of the dimensionless Helmholtz
   free energy to the inverse reduced temperature at constant reduced
   density.  */
double
if97i_t3_phi_tau_tau (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, delta, I(i));
      prod (tem, J(i) * (J(i) - 1), 1);
      prod (tem, tau, (J(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second derivative of the dimensionless Helmholtz free
   energy to the reduced density and the inverse reduced temperature.  */
double
if97i_t3_phi_delta_tau (double delta, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 1; i < 40; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, delta, (I(i) - 1));
      prod (tem, J(i), 1);
      prod (tem, tau, (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Calculate all properties of water at density RHO and temperature T.
   The result is returned in the variable PROP.

   Return value is the address of PROP.  */
struct if97 *
if97i_t3 (struct if97 *prop, double rho, double t)
{
  double delta, tau;
  double phi, phi_delta, phi_tau;
  double phi_delta_delta, phi_tau_tau, phi_delta_tau;
  double num, den, p, h, u, s, cp, cv, w;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi = if97i_t3_phi (delta, tau);
  phi_delta = if97i_t3_phi_delta (delta, tau);
  phi_tau = if97i_t3_phi_tau (delta, tau);
  phi_delta_delta = if97i_t3_phi_delta_delta (delta, tau);
  phi_tau_tau = if97i_t3_phi_tau_tau (delta, tau);
  phi_delta_tau = if97i_t3_phi_delta_tau (delta, tau);

  num = sq (delta * (phi_delta - tau * phi_delta_tau));
  den = delta * (2.0 * phi_delta + delta * phi_delta_delta);

  p = delta * phi_delta;
  u = tau * phi_tau;
  h = u + p;
  s = u - phi;
  cv = - sq (tau) * phi_tau_tau;
  cp = cv + num / den;
  w = den + num / cv;

  prop->if_p = p * rho * R * t;
  prop->if_t = t;
  prop->if_v = 1.0 / rho;
  prop->if_h = R * t * h;
  prop->if_u = R * t * u;
  prop->if_s = R * s;
  prop->if_cp = R * cp;
  prop->if_cv = R * cv;
  prop->if_w = sqrt (R * t * w);

  return prop;
}


/* Return the pressure at density RHO and temperature T.  */
double
if97i_t3_p (double rho, double t)
{
  double delta, tau;
  double phi_delta;
  double p;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_delta = if97i_t3_phi_delta (delta, tau);

  p = delta * phi_delta;

  return p * rho * R * t;
}


/* Return the specific volume at density RHO and temperature T.  */
double
if97i_t3_v (double rho, double t)
{
  return 1.0 / rho;
}


/* Return the specific enthalpy at density RHO and temperature T.  */
double
if97i_t3_h (double rho, double t)
{
  double delta, tau;
  double phi_delta, phi_tau;
  double p, h, u;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_delta = if97i_t3_phi_delta (delta, tau);
  phi_tau = if97i_t3_phi_tau (delta, tau);

  p = delta * phi_delta;
  u = tau * phi_tau;
  h = u + p;

  return R * t * h;
}


/* Return the specific internal energy at density RHO and temperature T.  */
double
if97i_t3_u (double rho, double t)
{
  double delta, tau;
  double phi_tau;
  double u;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_tau = if97i_t3_phi_tau (delta, tau);

  u = tau * phi_tau;

  return R * t * u;
}


/* Return the specific entropy at density RHO and temperature T.  */
double
if97i_t3_s (double rho, double t)
{
  double delta, tau;
  double phi, phi_tau;
  double u, s;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi = if97i_t3_phi (delta, tau);
  phi_tau = if97i_t3_phi_tau (delta, tau);

  u = tau * phi_tau;
  s = u - phi;

  return R * s;
}


/* Return the specific isobaric heat capacity at density RHO and
   temperature T.  */
double
if97i_t3_cp (double rho, double t)
{
  double delta, tau;
  double phi_delta;
  double phi_delta_delta, phi_tau_tau, phi_delta_tau;
  double num, den, cp, cv;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_delta = if97i_t3_phi_delta (delta, tau);
  phi_delta_delta = if97i_t3_phi_delta_delta (delta, tau);
  phi_tau_tau = if97i_t3_phi_tau_tau (delta, tau);
  phi_delta_tau = if97i_t3_phi_delta_tau (delta, tau);

  num = sq (delta * (phi_delta - tau * phi_delta_tau));
  den = delta * (2.0 * phi_delta + delta * phi_delta_delta);

  cv = - sq (tau) * phi_tau_tau;
  cp = cv + num / den;

  return R * cp;
}


/* Return the specific isochoric heat capacity at density RHO and
   temperature T.  */
double
if97i_t3_cv (double rho, double t)
{
  double delta, tau;
  double phi_tau_tau;
  double cv;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_tau_tau = if97i_t3_phi_tau_tau (delta, tau);

  cv = - sq (tau) * phi_tau_tau;

  return R * cv;
}


/* Return the speed of sound at density RHO and temperature T.  */
double
if97i_t3_w (double rho, double t)
{
  double delta, tau;
  double phi_delta;
  double phi_delta_delta, phi_tau_tau, phi_delta_tau;
  double num, den, cv, w;

  delta = if97i_t3_delta (rho);
  tau = if97i_t3_tau (t);

  phi_delta = if97i_t3_phi_delta (delta, tau);
  phi_delta_delta = if97i_t3_phi_delta_delta (delta, tau);
  phi_tau_tau = if97i_t3_phi_tau_tau (delta, tau);
  phi_delta_tau = if97i_t3_phi_delta_tau (delta, tau);

  num = sq (delta * (phi_delta - tau * phi_delta_tau));
  den = delta * (2.0 * phi_delta + delta * phi_delta_delta);

  cv = - sq (tau) * phi_tau_tau;
  w = den + num / cv;

  return sqrt (R * t * w);
}
