/* t4.c --- equations for region 4 (saturation line).

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-errno.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.34: Coefficients of equations (3.22) to (3.24).  */
struct t334
  {
    /* Index.  */
    int i;

    /* Coefficient.  */
    double n;
  };

static const struct t334 t334[] =
  {
    { 1,  0.11670521452767E+04},
    { 2, -0.72421316703206E+06},
    { 3, -0.17073846940092E+02},
    { 4,  0.12020824702470E+05},
    { 5, -0.32325550322333E+07},
    { 6,  0.14915108613530E+02},
    { 7, -0.48232657361591E+04},
    { 8,  0.40511340542057E+06},
    { 9, -0.23855557567849E+00},
    {10,  0.65017534844798E+03},
  };

#define n(i) t334[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference temperature.  */
#define t_star 1.0


/* Return the saturation pressure corresponding to the saturation
   temperature T.  */
double
if97i_p_s (double t)
{
  double theta, D;
  double A, B, C;

  theta = t / t_star + n(8) / (t / t_star - n(9));

  A = n(1) + theta * (n(0) + theta);
  B = n(4) + theta * (n(3) + theta * n(2));
  C = n(7) + theta * (n(6) + theta * n(5));

  D = 2.0 * C / (sqrt (B * B - 4.0 * A * C) - B);

  return p_star * D * D * D * D;
}


/* Return the saturation temperature corresponding to the saturation
   pressure P.  */
double
if97i_t_s (double p)
{
  double beta, D;
  double E, F, G;

  beta = sqrt (sqrt (p / p_star));

  E = n(5) + beta * (n(2) + beta);
  F = n(6) + beta * (n(3) + beta * n(0));
  G = n(7) + beta * (n(4) + beta * n(1));

  D = -2.0 * G / (sqrt (F * F - 4.0 * E * G) + F);

  return t_star * (n(9) + D - sqrt (sq (n(9) + D) - 4.0 * (n(8) + n(9) * D))) / 2.0;
}


/* Return the saturation pressure corresponding to the saturation
   temperature T.

   This function signals a domain error if the temperature T is not
   in the range [273.15 647.096] K.  */
double
if97_p_s (double t)
{
  if (t < T0 || t > TC)
    set_errno (EDOM);

  return if97i_p_s (t);
}


/* Return the saturation temperature corresponding to the saturation
   pressure P.

   This function signals a domain error if the pressure P is not in
   the range [611.213 22.064E+6] Pa.  */
double
if97_t_s (double p)
{
  if (p < P0 || p > PC)
    set_errno (EDOM);

  return if97i_t_s (p);
}


/* Return non-zero if pressure P and temperature T define saturation
   conditions inside the uncertainties of the equations.  */
int
if97_saturation_p (double p, double t)
{
  double p_s;

  if (p < PT || p > PC)
    return 0;

  if (t < TT || t > TC)
    return 0;

  p_s = if97i_p_s (t);
  if (fabs (p_s - p) / p > 0.05E-2)
    return 0;

  return 1;
}
