/* t5.c --- basic equation for region 5.

   Copyright (C) 2001, 2002 Ralph Schleicher

   Author: Ralph Schleicher <rs@nunatak.allgaeu.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307, USA.  */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <c-stand.h>
#include <c-math.h>

#include "if97.h"
#include "if97i.h"
#include "local.h"

/* Table 3.27: Coefficients and exponents of equation (3.26).  */
struct t327
  {
    /* Index.  */
    int i;

    /* Exponent.  */
    double J;

    /* Coefficient.  */
    double n;
  };

static const struct t327 t327[] =
  {
    {1,  0.0, -0.13179983674201E+02},
    {2,  1.0,  0.68540841634434E+01},
    {3, -3.0, -0.24805148933466E-01},
    {4, -2.0,  0.36901534980333E+00},
    {5, -1.0, -0.31161318213925E+01},
    {6,  2.0,  0.32961626538917E+00},
  };

#define J0(i) t327[i].J
#define n0(i) t327[i].n

/* Table 3.28: Coefficients and exponents of equation (3.27).  */
struct t328
  {
    /* Index.  */
    int i;

    /* Exponents.  */
    double I, J;

    /* Coefficient.  */
    double n;
  };

static const struct t328 t328[] =
  {
    { 1, 1.0, 0.0, -0.12563183589592E-03},
    { 2, 1.0, 1.0,  0.21774678714571E-02},
    { 3, 1.0, 3.0, -0.45942820899910E-02},
    { 4, 2.0, 9.0, -0.39724828359569E-05},
    { 5, 3.0, 3.0,  0.12919228289784E-06},
  };

#define I(i) t328[i].I
#define J(i) t328[i].J
#define n(i) t328[i].n

/* Reference pressure.  */
#define p_star 1.0E+6

/* Reference temperature.  */
#define t_star 1000.0


/* Return the reference pressure.  */
double
if97i_t5_p_star (void)
{
  return p_star;
}


/* Return the reference temperature.  */
double
if97i_t5_t_star (void)
{
  return t_star;
}


/* Return the reduced pressure.  */
double
if97i_t5_pi (double p)
{
  return p / p_star;
}


/* Return the inverse reduced temperature.  */
double
if97i_t5_tau (double t)
{
  return t_star / t;
}


/* Return the ideal-gas part of the dimensionless Gibbs free energy.  */
double
if97i_t5_gamma_0 (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 6; ++i)
    {
      double tem = n0(i);

      prod (tem, tau, J0(i));

      sum += tem;
    }

  return log (pi) + sum;
}


/* Return the first partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t5_gamma_0_pi (double pi, double tau __attribute__ ((unused)))
{
  return 1.0 / pi;
}


/* Return the first partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t5_gamma_0_tau (double pi __attribute__ ((unused)), double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 6; ++i)
    {
      double tem = n0(i);

      prod (tem, J0(i), 1);
      prod (tem, tau, (J0(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t5_gamma_0_pi_pi (double pi, double tau __attribute__ ((unused)))
{
  return -1.0 / (pi * pi);
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t5_gamma_0_tau_tau (double pi __attribute__ ((unused)), double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 6; ++i)
    {
      double tem = n0(i);

      prod (tem, J0(i) * (J0(i) - 1), 1);
      prod (tem, tau, (J0(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the ideal-gas part of the
   dimensionless Gibbs free energy to the reduced pressure and the
   inverse reduced temperature.  */
double
if97i_t5_gamma_0_pi_tau (double pi __attribute__ ((unused)), double tau __attribute__ ((unused)))
{
  return 0.0;
}


/* Return the residual part of the dimensionless Gibbs free energy.  */
double
if97i_t5_gamma_r (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return sum;
}


/* Return the first partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t5_gamma_r_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, pi, (I(i) - 1));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return sum;
}


/* Return the first partial derivative of the residual part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t5_gamma_r_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, J(i), 1);
      prod (tem, tau, (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure at constant
   inverse reduced temperature.  */
double
if97i_t5_gamma_r_pi_pi (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, I(i) * (I(i) - 1), 1);
      prod (tem, pi, (I(i) - 2));
      prod (tem, tau, J(i));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the inverse reduced temperature
   at constant reduced pressure.  */
double
if97i_t5_gamma_r_tau_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, pi, I(i));
      prod (tem, J(i) * (J(i) - 1), 1);
      prod (tem, tau, (J(i) - 2));

      sum += tem;
    }

  return sum;
}


/* Return the second partial derivative of the residual part of the
   dimensionless Gibbs free energy to the reduced pressure and the
   inverse reduced temperature.  */
double
if97i_t5_gamma_r_pi_tau (double pi, double tau)
{
  double sum = 0.0;
  int i;

  for (i = 0; i < 5; ++i)
    {
      double tem = n(i);

      prod (tem, I(i), 1);
      prod (tem, pi, (I(i) - 1));
      prod (tem, J(i), 1);
      prod (tem, tau, (J(i) - 1));

      sum += tem;
    }

  return sum;
}


/* Calculate all properties at pressure P and temperature T.
   The result is returned in the variable PROP.

   Return value is the address of PROP.  */
struct if97 *
if97i_t5 (struct if97 *prop, double p, double t)
{
  double pi, tau;
  double gamma_0, gamma_0_pi, gamma_0_tau;
  double gamma_0_pi_pi, gamma_0_tau_tau, gamma_0_pi_tau;
  double gamma_r, gamma_r_pi, gamma_r_tau;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, v, h, u, s, cp, cv, w;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0 = if97i_t5_gamma_0 (pi, tau);
  gamma_0_pi = if97i_t5_gamma_0_pi (pi, tau);
  gamma_0_tau = if97i_t5_gamma_0_tau (pi, tau);
  gamma_0_pi_pi = if97i_t5_gamma_0_pi_pi (pi, tau);
  gamma_0_tau_tau = if97i_t5_gamma_0_tau_tau (pi, tau);
  gamma_0_pi_tau = if97i_t5_gamma_0_pi_tau (pi, tau);

  gamma_r = if97i_t5_gamma_r (pi, tau);
  gamma_r_pi = if97i_t5_gamma_r_pi (pi, tau);
  gamma_r_tau = if97i_t5_gamma_r_tau (pi, tau);
  gamma_r_pi_pi = if97i_t5_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t5_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t5_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * gamma_r_pi - tau * pi * gamma_r_pi_tau);
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  v = pi * (gamma_0_pi + gamma_r_pi);
  h = tau * (gamma_0_tau + gamma_r_tau);
  u = h - v;
  s = h - (gamma_0 + gamma_r);
  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  cv = cp - num / den;
  w = (1.0 + pi * gamma_r_pi * (2.0 + pi * gamma_r_pi)) / (den - num / cp);

  prop->if_p = p;
  prop->if_t = t;
  prop->if_v = R * t / p * v;
  prop->if_h = R * t * h;
  prop->if_u = R * t * u;
  prop->if_s = R * s;
  prop->if_cp = R * cp;
  prop->if_cv = R * cv;
  prop->if_w = sqrt (R * t * w);

  return prop;
}


/* Return the specific volume at pressure P and temperature T.  */
double
if97i_t5_v (double p, double t)
{
  double pi, tau;
  double gamma_0_pi;
  double gamma_r_pi;
  double v;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_pi = if97i_t5_gamma_0_pi (pi, tau);
  gamma_r_pi = if97i_t5_gamma_r_pi (pi, tau);

  v = pi * (gamma_0_pi + gamma_r_pi);

  return R * t / p * v;
}


/* Return the specific enthalpy at pressure P and temperature T.  */
double
if97i_t5_h (double p, double t)
{
  double pi, tau;
  double gamma_0_tau;
  double gamma_r_tau;
  double h;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_tau = if97i_t5_gamma_0_tau (pi, tau);
  gamma_r_tau = if97i_t5_gamma_r_tau (pi, tau);

  h = tau * (gamma_0_tau + gamma_r_tau);

  return R * t * h;
}


/* Return the specific internal energy at pressure P and temperature T.  */
double
if97i_t5_u (double p, double t)
{
  double pi, tau;
  double gamma_0_pi, gamma_0_tau;
  double gamma_r_pi, gamma_r_tau;
  double v, h, u;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_pi = if97i_t5_gamma_0_pi (pi, tau);
  gamma_0_tau = if97i_t5_gamma_0_tau (pi, tau);
  gamma_r_pi = if97i_t5_gamma_r_pi (pi, tau);
  gamma_r_tau = if97i_t5_gamma_r_tau (pi, tau);

  v = pi * (gamma_0_pi + gamma_r_pi);
  h = tau * (gamma_0_tau + gamma_r_tau);
  u = h - v;

  return R * t * u;
}


/* Return the specific entropy at pressure P and temperature T.  */
double
if97i_t5_s (double p, double t)
{
  double pi, tau;
  double gamma_0, gamma_0_tau;
  double gamma_r, gamma_r_tau;
  double h, s;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0 = if97i_t5_gamma_0 (pi, tau);
  gamma_0_tau = if97i_t5_gamma_0_tau (pi, tau);
  gamma_r = if97i_t5_gamma_r (pi, tau);
  gamma_r_tau = if97i_t5_gamma_r_tau (pi, tau);

  h = tau * (gamma_0_tau + gamma_r_tau);
  s = h - (gamma_0 + gamma_r);

  return R * s;
}


/* Return the specific isobaric heat capacity at pressure P and
   temperature T.  */
double
if97i_t5_cp (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_tau_tau;
  double cp;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_tau_tau = if97i_t5_gamma_0_tau_tau (pi, tau);
  gamma_r_tau_tau = if97i_t5_gamma_r_tau_tau (pi, tau);

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);

  return R * cp;
}


/* Return the specific isochoric heat capacity at pressure P and
   temperature T.  */
double
if97i_t5_cv (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_pi;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, cp, cv;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_tau_tau = if97i_t5_gamma_0_tau_tau (pi, tau);
  gamma_r_pi = if97i_t5_gamma_r_pi (pi, tau);
  gamma_r_pi_pi = if97i_t5_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t5_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t5_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * gamma_r_pi - tau * pi * gamma_r_pi_tau);
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  cv = cp - num / den;

  return R * cv;
}


/* Return the speed of sound at pressure P and temperature T.  */
double
if97i_t5_w (double p, double t)
{
  double pi, tau;
  double gamma_0_tau_tau;
  double gamma_r_pi;
  double gamma_r_pi_pi, gamma_r_tau_tau, gamma_r_pi_tau;
  double num, den, cp, w;

  pi = if97i_t5_pi (p);
  tau = if97i_t5_tau (t);

  gamma_0_tau_tau = if97i_t5_gamma_0_tau_tau (pi, tau);
  gamma_r_pi = if97i_t5_gamma_r_pi (pi, tau);
  gamma_r_pi_pi = if97i_t5_gamma_r_pi_pi (pi, tau);
  gamma_r_tau_tau = if97i_t5_gamma_r_tau_tau (pi, tau);
  gamma_r_pi_tau = if97i_t5_gamma_r_pi_tau (pi, tau);

  num = sq (1.0 + pi * gamma_r_pi - tau * pi * gamma_r_pi_tau);
  den = 1.0 - sq (pi) * gamma_r_pi_pi;

  cp = - sq (tau) * (gamma_0_tau_tau + gamma_r_tau_tau);
  w = (1.0 + pi * gamma_r_pi * (2.0 + pi * gamma_r_pi)) / (den - num / cp);

  return sqrt (R * t * w);
}
