;;; hello.lisp --- print a greeting message.

;; Copyright (C) 2011, 2012 Ralph Schleicher

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;;
;;    * Redistributions of source code must retain the above copyright
;;      notice, this list of conditions and the following disclaimer.
;;
;;    * Redistributions in binary form must reproduce the above copyright
;;      notice, this list of conditions and the following disclaimer in
;;      the documentation and/or other materials provided with the
;;      distribution.
;;
;;    * The name of the author may not be used to endorse or promote
;;      products derived from this software without specific prior
;;      written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS
;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;; ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
;; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
;; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
;; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
;; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
;; IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;; POSSIBILITY OF SUCH DAMAGE.

;;; Code:

(in-package :common-lisp-user)

(defpackage :hello
  (:use :common-lisp
	:iterate
	:rs-lisp
	:rs-app
	:rs-getopt))

(in-package :hello)

;;;; Implementation.

(export '*greeting*)
(defparameter *greeting* "Hello, world!"
  "The greeting message (a string).")

(export 'hello)
(defun hello ()
  "The greeting program."
  ;; Display the greeting message.
  (fresh-line)
  (princ *greeting*)
  (fresh-line))

;;;; Standalone application.

(defconst +PROGRAM+ "hello"
  "Official name of the program.")

(defconst +VERSION+ "1.0"
  "Version number of the program.")

(defconst +ADDRESS+ (format nil "<~A@~A>" "rs" "ralph-schleicher.de")
  "Mail address or URL for reporting bugs.")

(defparameter *show-version* nil
  "Non-null means to print the version number.")

(defparameter *show-help* nil
  "Non-null means to print the help text.")

(defun show-version (&optional (stream *standard-output*))
  "Display version number information."
  (format stream "~
~A ~A

Copyright (C) 2011, 2012 Ralph Schleicher

This program is free software and distributed under the modified
BSD License.  There is NO warranty; not even for MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.~%"
	  +PROGRAM+ +VERSION+))

(defun show-help (&optional (stream *standard-output*))
  "Display help text."
  (format stream "~
Usage: ~A [OPTION...]

Print a friendly, customizable greeting.

  -g, --greeting=TEXT   Use TEXT as the greeting message.
  -t, --traditional     Use a traditional greeting message.
  --version             Display version number information.
  --help                Display this help text.

Report bugs to ~A~%"
	  (program-invocation-short-name) +ADDRESS+))

(export 'main)
(defun main (&rest arguments)
  "Program entry point."
  (declare (ignore arguments))
  ;; This is a standalone application and
  ;; not an interactive Lisp process.
  (standalone-program)
  ;; Get options and arguments.
  (let (opt)
    (case 'second
      (first
       ;; Combine a simple options specification with an explicit
       ;; loop for option handling.  This resembles the C language
       ;; programming pattern.
       (setf opt (make-getopt '(("greeting" #\g
				 :argument :required)
				("traditional" #\t)
				"version"
				"help")))
       (iter (case (getopt opt)
	       ((nil)
		(finish))
	       (("greeting" #\g)
		(setf *greeting* (optarg opt)))
	       (("traditional" #\t)
		(setf *greeting* "hello, world"))
	       ("version"
		(setf *show-version* t))
	       ("help"
		(setf *show-help* t))
	       (t
		(die "Try '~A --help' for more information"
		     (program-invocation-name))))))
      (second
       ;; Add ACTION keywords to the options specification for
       ;; implicit option handling.  Also add the HELP keyword
       ;; so that `show-help-hint-and-die' knowns the name of
       ;; the help option.
       (setf opt (make-getopt `(("greeting" #\g
				 :argument :required
				 :action *greeting*)
				("traditional" #\t
				 :action ,(lambda (key value)
					    (declare (ignore key value))
					    (setf *greeting* "hello, world")))
				("version"
				 :action *show-version*)
				("help"
				 :action *show-help*))
			      :help "--help"))
       (when (getopt opt)
	 (show-help-hint-and-die opt))))
    ;; Check for help.
    (when (or *show-version* *show-help*)
      (fresh-line)
      (when *show-version*
	(show-version))
      (when *show-help*
	(when *show-version*
	  (terpri) (terpri))
	(show-help))
      (exit-success))
    ;; Check remaining arguments.
    (when (remaining-arguments opt)
      (say "too many command line arguments")
      (show-help-hint-and-die opt)))
  ;; Run the actual program.
  (hello))

;;; hello.lisp ends here
