;;; temp-dir.lisp --- temporary files and directories.

;; Copyright (C) 2012 Ralph Schleicher

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;;
;;    * Redistributions of source code must retain the above copyright
;;      notice, this list of conditions and the following disclaimer.
;;
;;    * Redistributions in binary form must reproduce the above copyright
;;      notice, this list of conditions and the following disclaimer in
;;      the documentation and/or other materials provided with the
;;      distribution.
;;
;;    * The name of the author may not be used to endorse or promote
;;      products derived from this software without specific prior
;;      written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS
;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;; ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
;; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
;; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
;; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
;; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
;; IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;; POSSIBILITY OF SUCH DAMAGE.

;;; Code:

(in-package :rs-app)

(defun temp-dir (directory)
  (cond ((null directory)
	 *default-pathname-defaults*)
	((eq directory t)
	 (pathname-as-directory
	  (or (environment-variable "TMPDIR")
	      (environment-variable "TMP")
	      #+unix
	      "/tmp"
	      #+windows
	      (or (environment-variable "TEMP")
		  "C:\\Temp")
	      #-(or unix windows)
	      (fix-me 'temp-dir))))
	((stringp directory)
	 (pathname-as-directory directory))
	(t
	 (pathname directory))))

(export 'temporary-file-name)
(defun temporary-file-name (&key (prefix "temp") directory)
  (declare (type string prefix))
  (let ((defaults (temp-dir directory)))
    (iter (for tem = (make-pathname
		      :name (concatenate 'string prefix (random-string 6))
		      :defaults defaults))
	  (unless (file-exists-p tem)
	    (return tem)))))

(export 'temporary-file)
(defun temporary-file (&key (prefix "temp") directory (direction :output) (element-type 'character) (external-format :default))
  "Create a unique file."
  (declare (type string prefix))
  (let ((defaults (temp-dir directory)))
    (iter (for stream = (open (make-pathname
			       :name (concatenate 'string prefix (random-string 6))
			       :defaults defaults)
			      :direction direction
			      :element-type element-type
			      :if-exists nil
			      :if-does-not-exist :create
			      :external-format external-format))
	  (when (not (null stream))
	    (return stream)))))

(export 'with-temporary-file)
(defmacro with-temporary-file ((stream &rest arg) &body body)
  (let ((temp (gensym "temp")))
    `(let* ((,stream (temporary-file ,@arg))
	    (,temp (pathname ,stream)))
       (unwind-protect
	    (progn ,@body)
	 (when (ignore-errors (open-stream-p ,stream))
	   (close ,stream))
	 (ignore-errors (delete-file ,temp))))))

(export 'temporary-directory)
(defun temporary-directory (&key (prefix "temp") directory)
  (declare (type string prefix))
  (let ((defaults (temp-dir directory)))
    (let (dir created)
      (iter (for tem = (make-pathname
			:name (concatenate 'string prefix (random-string 6))
			:defaults defaults))
	    (multiple-value-setq (dir created)
	      (ensure-directories-exist (pathname-as-directory tem)))
	    (when (not (null created))
	      (return dir))))))

(export 'with-temporary-directory)
(defmacro with-temporary-directory ((var &rest arg) &body body)
  `(let ((,var (temporary-directory ,@arg)))
     (unwind-protect
	  (progn ,@body)
       (ignore-errors (delete-directory-and-files ,var)))))

;;; temp-dir.lisp ends here
