;;; quantities.lisp --- quantities and units.

;; Copyright (C) 2012 Ralph Schleicher

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;;
;;    * Redistributions of source code must retain the above copyright
;;      notice, this list of conditions and the following disclaimer.
;;
;;    * Redistributions in binary form must reproduce the above copyright
;;      notice, this list of conditions and the following disclaimer in
;;      the documentation and/or other materials provided with the
;;      distribution.
;;
;;    * The name of the author may not be used to endorse or promote
;;      products derived from this software without specific prior
;;      written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS
;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;; ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
;; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
;; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
;; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
;; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
;; IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;; POSSIBILITY OF SUCH DAMAGE.

;;; Code:

(in-package :rs-lisp)

(export 'radian-from-degree)
(defun radian-from-degree (deg &optional (min 0 sexagesimal) (s 0) (sign 1))
  "Convert a plane angle from degree to radian.

First argument DEG is the number of arc degrees.
Optional second argument MIN is the number of arc minutes.
Optional third argument S is the number of arc seconds.
Optional fourth argument SIGN specifies the sign.

Value is the corresponding angle given in radian.

If two or more arguments are supplied, then MIN and S are sexagesimal
subdivisions of an arc degree.  In this case, DEG, MIN, and S have to
be non-negative numbers and the sign of SIGN, after conversion into a
floating-point number, determines the sign of the angle."
  (declare (type real deg min s sign))
  (if (not sexagesimal)
      (* (/ deg 180) pi)
    (locally
	(declare (type (real 0) deg min s))
      (float-sign (float sign) (* (/ (+ (/ (+ (/ s 60) min) 60) deg) 180) pi)))))

(export 'degree-from-radian)
(defun degree-from-radian (rad &key sexagesimal)
  "Convert a plane angle from radian to degree.

First argument RAD is the angle given in radian.
If keyword argument SEXAGESIMAL is true, subdivide the return value
 in arc degrees, arc minutes, and arc seconds.  The number of arc
 seconds may be a floating point number with fractions of a second.
 Fourth value is either plus one or minus one specifying the sign
 of the angle.

Value is the corresponding angle given in degree."
  (declare (type real rad))
  (if (not sexagesimal)
      (* (/ rad pi) 180)
    (let (sign deg min s f)
      (setf sign (if (minusp rad) -1 +1))
      (multiple-value-setq (deg f)
	(floor (* (/ (abs rad) pi) 180)))
      (multiple-value-setq (min f)
	(floor (* f 60)))
      (multiple-value-setq (s f)
	(floor (* f 60)))
      (values deg min (if (zerop f) s (+ s f)) sign))))

;;; quantities.lisp ends here
