/* textps.c -- convert ordinary text to PostScript.

   Copyright (C) 1996, 1997 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <sys/types.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#else
extern int waitpid ();
#endif
#ifndef WAIT_ANY
#define WAIT_ANY (-1)
#endif
#ifndef WIFEXITED
#define WIFEXITED(status) (((status) & 0xff) == 0)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(status) ((unsigned) (status) >> 8)
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#else /* not STDC_HEADERS */
#include <varargs.h>
extern char *getenv ();
#if ! HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* not HAVE_STRCHR */
extern char *strchr ();
extern char *strrchr ();
#if ! HAVE_STRERROR
extern char *sys_errlist[];
#define strerror(error) sys_errlist[error]
#endif /* not HAVE_STRERROR */
#endif /* not STDC_HEADERS */

#include <stdio.h>
#if HAVE_FCNTL_H
#include <fcntl.h>
#else /* not HAVE_FCNTL_H */
extern int fcntl ();
#ifndef FD_CLOEXEC
#define FD_CLOEXEC 1
#endif /* not FD_CLOEXEC */
#endif /* not HAVE_FCNTL_H */
#if HAVE_UNISTD_H
#include <unistd.h>
#else /* not HAVE_UNISTD_H */
extern int access ();
#ifndef R_OK
#define R_OK 4
#endif /* not R_OK */
extern int dup ();
extern int dup2 ();
extern int pipe ();
extern int fork ();
extern int execv ();
extern int execvp ();
extern int _exit ();
#endif /* not HAVE_UNISTD_H */

#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <time.h>

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#endif
#ifndef EXIT_FAILURE
#define EXIT_FAILURE 1
#endif

#if STDC_HEADERS
#define VA_LIST				...
#define VA_DECL				...
#define VA_START(args, last)		va_start (args, last)
#define EXFUN(name, list)		name list
#define DEFUN(name, list, arg)		name (arg)
#define DEFUN_VAR(name, list, arg) 	name (arg)
#define DEFUN_VOID(name)		name (void)
#define AND				,
#else /* not STDC_HEADERS */
#define VA_LIST				va_alist
#define VA_DECL				va_dcl
#define VA_START(args, last)		va_start (args)
#define EXFUN(name, list)		name ()
#define DEFUN(name, list, arg)		name list arg;
#define DEFUN_VAR(name, list, arg)	name list arg
#define DEFUN_VOID(name)		name ()
#define AND				;
#endif /* not STDC_HEADERS */

extern void EXFUN (freeargv, (char **vector));
extern char **EXFUN (buildargv, (char *input, char *argv0));

static void EXFUN (set_defaults, (void));
static void EXFUN (get_options, (int arg_count, char **arg_vec, const char *env));
static void EXFUN (prolog, (void));
static void EXFUN (trailer, (void));
static void EXFUN (begin_page, (void));
static void EXFUN (show_page, (void));
static void EXFUN (input_loop, (void));
static void EXFUN (add_char, (int code, int face));
static void EXFUN (print_line, (int force));
static char *EXFUN (format_number, (char *string, double number));
static int EXFUN (scan_dimen, (const char *string, double *number, int not_used));
static int EXFUN (private, (int handle));
static void EXFUN (die, (const char *format, VA_LIST));


static char *invocation_name = "textps";
static char *program_name = "textps";
static char *version_string = "1.0.2";

/* Predefined paper formats.  The default media is `letter'.  This can be
   changed to DIN A4 if the macro `A4' is defined at compile time.  */

#ifndef A4
#define A4 0
#endif

#define IN(value) ((value) * 72.0)
#define MM(value) IN (value / 25.4)

struct media
  {
    char *name;
    int width;
    int height;
  };

static struct media media_vec[] =
  {
    {"letter", IN (8.5), IN (11.0)},
    {"legal", IN (8.5), IN (14.0)},
    {"ledger", IN (17.0), IN (11.0)},
    {"tabloid", IN (11.0), IN (17.0)},
    {"11x17", IN (11.0), IN (17.0)},
    {"folio", IN (8.5), IN (13.0)},
    {"executive", IN (7.5), IN (10.0)},
    {"note", IN (7.5), IN (10.0)},
    {"statement", IN (5.5), IN (8.5)},
    {"halfletter", IN (5.5), IN (8.5)},
    {"10x14", IN (10.0), IN (14.0)},
    {"A3", MM (297.0), MM (420.0)},
    {"A4", MM (210.0), MM (297.0)},
    {"A5", MM (148.0), MM (210.0)},
    {"A6", MM (106.0), MM (148.0)},
    {"B3", MM (354.0), MM (500.0)},
    {"B4", MM (250.0), MM (354.0)},
    {"B5", MM (177.0), MM (250.0)},
    {"B6", MM (125.0), MM (177.0)},
  };

static int media_count = sizeof (media_vec) / sizeof (media_vec[0]);

#if A4
static double paper_width = MM (210.0);
static double paper_height = MM (297.0);
#else /* not A4 */
static double paper_width = IN (8.5);
static double paper_height = IN (11.0);
#endif /* not A4 */

/* Character encoding tables.  The default character encoding is ASCII.
   This can be changed to ISO 8859-1 if the macro `LATIN1' is defined at
   compile time.  */

#ifndef LATIN1
#define LATIN1 0
#endif

static char *ascii_encoding[256] =
  {
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    "space", "exclam", "quotedbl", "numbersign",
    "dollar", "percent", "ampersand", "quoteright",
    "parenleft", "parenright", "asterisk", "plus",
    "comma", "hyphen", "period", "slash",

    "zero", "one", "two", "three",
    "four", "five", "six", "seven",
    "eight", "nine", "colon", "semicolon",
    "less", "equal", "greater", "question",

    "at", "A", "B", "C",
    "D", "E", "F", "G",
    "H", "I", "J", "K",
    "L", "M", "N", "O",

    "P", "Q", "R", "S",
    "T", "U", "V", "W",
    "X", "Y", "Z", "bracketleft",
    "backslash", "bracketright", "asciicircum", "underscore",

    "quoteleft", "a", "b", "c",
    "d", "e", "f", "g",
    "h", "i", "j", "k",
    "l", "m", "n", "o",

    "p", "q", "r", "s",
    "t", "u", "v", "w",
    "x", "y", "z", "braceleft",
    "bar", "braceright", "asciitilde", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
  };

static char *standard_encoding[256] =
  {
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    "space", "exclam", "quotedbl", "numbersign",
    "dollar", "percent", "ampersand", "quoteright",
    "parenleft", "parenright", "asterisk", "plus",
    "comma", "hyphen", "period", "slash",

    "zero", "one", "two", "three",
    "four", "five", "six", "seven",
    "eight", "nine", "colon", "semicolon",
    "less", "equal", "greater", "question",

    "at", "A", "B", "C",
    "D", "E", "F", "G",
    "H", "I", "J", "K",
    "L", "M", "N", "O",

    "P", "Q", "R", "S",
    "T", "U", "V", "W",
    "X", "Y", "Z", "bracketleft",
    "backslash", "bracketright", "asciicircum", "underscore",

    "quoteleft", "a", "b", "c",
    "d", "e", "f", "g",
    "h", "i", "j", "k",
    "l", "m", "n", "o",

    "p", "q", "r", "s",
    "t", "u", "v", "w",
    "x", "y", "z", "braceleft",
    "bar", "braceright", "asciitilde", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", "exclamdown", "cent", "sterling",
    "fraction", "yen", "florin", "section",
    "currency", "quotesingle", "quotedblleft", "guillemotleft",
    "guilsinglleft", "guilsinglright", "fi", "fl",

    ".notdef", "endash", "dagger", "daggerdbl",
    "periodcentered", ".notdef", "paragraph", "bullet",
    "quotesinglbase", "quotedblbase", "quotedblright", "guillemotright",
    "ellipsis", "perthousand", ".notdef", "questiondown",

    ".notdef", "grave", "acute", "circumflex",
    "tilde", "macron", "breve", "dotaccent",
    "dieresis", ".notdef", "ring", "cedilla",
    ".notdef", "hungarumlaut", "ogonek", "caron",

    "emdash", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", "AE", ".notdef", "ordfeminine",
    ".notdef", ".notdef", ".notdef", ".notdef",
    "Lslash", "Oslash", "OE", "ordmasculine",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", "ae", ".notdef", ".notdef",
    ".notdef", "dotlessi", ".notdef", ".notdef",
    "lslash", "oslash", "oe", "germandbls",
    ".notdef", ".notdef", ".notdef", ".notdef",
  };

static char *iso_8859_1_encoding[256] =
  {
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    "space", "exclam", "quotedbl", "numbersign",
    "dollar", "percent", "ampersand", "quoteright",
    "parenleft", "parenright", "asterisk", "plus",
    "comma", "minus", "period", "slash",

    "zero", "one", "two", "three",
    "four", "five", "six", "seven",
    "eight", "nine", "colon", "semicolon",
    "less", "equal", "greater", "question",

    "at", "A", "B", "C",
    "D", "E", "F", "G",
    "H", "I", "J", "K",
    "L", "M", "N", "O",

    "P", "Q", "R", "S",
    "T", "U", "V", "W",
    "X", "Y", "Z", "bracketleft",
    "backslash", "bracketright", "asciicircum", "underscore",

    "quoteleft", "a", "b", "c",
    "d", "e", "f", "g",
    "h", "i", "j", "k",
    "l", "m", "n", "o",

    "p", "q", "r", "s",
    "t", "u", "v", "w",
    "x", "y", "z", "braceleft",
    "bar", "braceright", "asciitilde", ".notdef",

    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",
    ".notdef", ".notdef", ".notdef", ".notdef",

    "dotlessi", "grave", "acute", "circumflex",
    "tilde", "macron", "breve", "dotaccent",
    "dieresis", ".notdef", "ring", "cedilla",
    ".notdef", "hungarumlaut", "ogonek", "caron",

    "space", "exclamdown", "cent", "sterling",
    "currency", "yen", "brokenbar", "section",
    "dieresis", "copyright", "ordfeminine", "guillemotleft",
    "logicalnot", "hyphen", "registered", "macron",

    "degree", "plusminus", "twosuperior", "threesuperior",
    "acute", "mu", "paragraph", "periodcentered",
    "cedilla", "onesuperior", "ordmasculine", "guillemotright",
    "onequarter", "onehalf", "threequarters", "questiondown",

    "Agrave", "Aacute", "Acircumflex", "Atilde",
    "Adieresis", "Aring", "AE", "Ccedilla",
    "Egrave", "Eacute", "Ecircumflex", "Edieresis",
    "Igrave", "Iacute", "Icircumflex", "Idieresis",

    "Eth", "Ntilde", "Ograve", "Oacute",
    "Ocircumflex", "Otilde", "Odieresis", "multiply",
    "Oslash", "Ugrave", "Uacute", "Ucircumflex",
    "Udieresis", "Yacute", "Thorn", "germandbls",

    "agrave", "aacute", "acircumflex", "atilde",
    "adieresis", "aring", "ae", "ccedilla",
    "egrave", "eacute", "ecircumflex", "edieresis",
    "igrave", "iacute", "icircumflex", "idieresis",

    "eth", "ntilde", "ograve", "oacute",
    "ocircumflex", "otilde", "odieresis", "divide",
    "oslash", "ugrave", "uacute", "ucircumflex",
    "udieresis", "yacute", "thorn", "ydieresis",
  };

struct charset
  {
    char *name;
    char **enc;
  };

static struct charset charset_vec[] =
  {
    {"ascii", ascii_encoding},
    {"adobe", standard_encoding},
    {"latin1", iso_8859_1_encoding},
    {"iso1", iso_8859_1_encoding},
  };

static int charset_count = sizeof (charset_vec) / sizeof (charset_vec[0]);

#if LATIN1
static char **encoding_vec = iso_8859_1_encoding;
#else /* not LATIN1 */
static char **encoding_vec = ascii_encoding;
#endif /* not LATIN1 */
static int encoding_count = 256;

/* The font is currently hard-wired in the code but this may change in the
   future.  */

#define ROMAN	0x0
#define BOLD	0x1
#define ITALIC	0x2

struct font
  {
    char *name;
    char *face[4];
  };

static struct font font_vec[] =
  {
    {"courier",
     {"Courier", "Courier-Bold", "Courier-Italic", "Courier-BoldItalic"}},
  };

static int font_count = sizeof (font_vec) / sizeof (font_vec[0]);

static char *face_abbrev[4] = {"R", "B", "I", "BI"};
static char **face_vec = font_vec[0].face;
static int face_count = 4;

static double font_size = 10.0;
static double font_pitch = 12.0;
static double char_width = 6.0;

/* Other customization variables.

   The value of the variable `bind_offset' will be added to `left_margin'
   if the page number if odd.

   `horiz_offset' and `vert_offset' may be used to adjust the upper left
   corner of the page.

   `top_skip' is the distance from the beginning of the page to the first
   baseline.  */

static double left_margin = 0.0;
static double top_margin = 0.0;
static double bind_offset = 0.0;
static double horiz_offset = 0.0;
static double vert_offset = 0.0;

static double top_skip = 8.5;
static double baseline_skip = 12.0;

static int tab_width = 8;
static int lines_per_page = 0;

static int no_bold = 0;
static int no_italic = 0;

static int turn_paper = 0;

static char *input_file = 0;
static char *output_file = 0;
static int control_d = 0;

struct child
  {
    char *command;
    char **arg_vec;
    int proc_id;
  };

static struct child *filter_vec = 0;
static int filter_count = 0;
static int filter_size = 0;

static int show_version = 0;
static int show_help = 0;

/* Short options are only used for common configuration tasks.  */

#define MEDIA		'm'
#define PAPER_WIDTH	256
#define PAPER_HEIGHT	257
#define HORIZ_OFFSET	258
#define VERT_OFFSET	259
#define FONT_SIZE	's'
#define FONT_PITCH	'p'
#define TOP_SKIP	260
#define BASELINE_SKIP	261
#define LEFT_MARGIN	'h'
#define TOP_MARGIN	'v'
#define BIND_OFFSET	'b'
#define ENCODING	'e'
#define TAB_WIDTH	't'
#define LINES_PER_PAGE	'l'
#define FILTER		'x'
#define NROFF		'n'
#define OUTPUT		'o'

static char *option_string = "m:s:p:h:v:b:e:t:l:x:no:";

static struct option option_vec[] =
  {
    {"media", required_argument, 0, MEDIA},
    {"paper-format", required_argument, 0, MEDIA},
    {"paper-width", required_argument, 0, PAPER_WIDTH},
    {"paper-height", required_argument, 0, PAPER_HEIGHT},
    {"landscape", no_argument, &turn_paper, 1},
    {"portrait", no_argument, &turn_paper, 0},
    {"horizontal-offset", required_argument, 0, HORIZ_OFFSET},
    {"vertical-offset", required_argument, 0, VERT_OFFSET},
    {"font-size", required_argument, 0, FONT_SIZE},
    {"font-pitch", required_argument, 0, FONT_PITCH},
    {"top-skip", required_argument, 0, TOP_SKIP},
    {"baseline-skip", required_argument, 0, BASELINE_SKIP},
    {"left-margin", required_argument, 0, LEFT_MARGIN},
    {"top-margin", required_argument, 0, TOP_MARGIN},
    {"binding-offset", required_argument, 0, BIND_OFFSET},
    {"encoding", required_argument, 0, ENCODING},
    {"tab-width", required_argument, 0, TAB_WIDTH},
    {"lines-per-page", required_argument, 0, LINES_PER_PAGE},
    {"filter", required_argument, 0, FILTER},
    {"nroff", no_argument, 0, NROFF},
    {"no-bold", no_argument, &no_bold, 1},
    {"no-italic", no_argument, &no_italic, 1},
    {"output", required_argument, 0, OUTPUT},
    {"end-of-file", no_argument, &control_d, 1},
    {"version", no_argument, &show_version, 1},
    {"help", no_argument, &show_help, 1},
    {0, 0, 0, 0},
  };

const char *help_string = "\
Usage:  textps [option] [file ...]\n\
\n\
Hardware Options\n\
-m NAME, --media=NAME, --paper-format=NAME\n\
	Select a predefined paper format.  Known paper formats are `letter',\n\
	`legal', `ledger', `tabloid', `11x17', `folio', `executive', `note',\n\
	`statement', `halfletter', `A3', `A4', `A5', `A6', `B3', `B4', `B5'\n\
	and `B6'.\n\
--paper-width=DIMEN\n\
--paper-height=DIMEN\n\
	Set the width and/or height of a non-standard paper format.\n\
--landscape\n\
	Rotate the sheet 90 degree clockwise.\n\
--portrait\n\
	Use the paper with it's original orientation.\n\
--horizontal-offset=DIMEN\n\
--vertical-offset=DIMEN\n\
	Adjust the print engine's origin to the upper left corner of the\n\
	physical page.\n\
\n\
Layout Options\n\
-s DIMEN, --font-size=DIMEN\n\
	Set the font size to DIMEN.\n\
-p NUMBER, --font-pitch=NUMBER\n\
	Adjust the font size for NUMBER characters per inch.\n\
--top-skip=DIMEN\n\
	The distance from the beginning of the page to the first baseline.\n\
--baseline-skip=DIMEN\n\
	Baselines are that much apart.\n\
-h DIMEN, --left-margin=DIMEN\n\
-v DIMEN, --top-margin=DIMEN\n\
	Set the width and height of the left and top margin.\n\
-b DIMEN, --binding-offset=DIMEN\n\
	The extra space added to the left side of odd pages.\n\
\n\
Formatting Options\n\
-e NAME, --encoding=NAME\n\
	Select a character set.  Known character encodings are `ascii',\n\
	`adobe' and `latin1'.  `iso1' is a synonym for `latin1'.\n\
-t NUMBER, --tab-width=NUMBER\n\
	A tab character is NUMBER characters wide.\n\
-l NUMBER, --lines-per-page=NUMBER\n\
	Break pages after NUMBER lines.\n\
-x COMMAND, --filter=COMMAND\n\
	Preprocess the input files with COMMAND.  Multiple input filters\n\
	are okay.\n\
-n, --nroff\n\
	Select nice formatting options for input files processed by nroff(1).\n\
--no-bold\n\
	Don't obey `bold' overstrike conventions.\n\
--no-italic\n\
	Don't obey `italic' overstrike conventions.\n\
\n\
Output Options\n\
-o FILE, --output=FILE\n\
	Write the PostScript image to the file FILE instead of writing it to\n\
	standard output.\n\
--end-of-file\n\
	Append an ASCII code 4 to the PostScript image.\n\
\n\
Miscellaneous Options\n\
--version\n\
	Print the version numberm then exit successfully.\n\
--help\n\
	Print this help, then exit successfully.\n";

/* Internal variables.  */

#define NUMBER_SIZE 512
#define NUMBER_FORMAT "%0.6f"

#define TIME_SIZE 256
#define TIME_FORMAT "%a %b %d %H:%M:%S %Z %Y"

static time_t curr_time;
static char time_string[TIME_SIZE];

static char *input_name = "(standard input)";
static char *output_name = "(standard output)";

static FILE *input_stream = stdin;
static FILE *output_stream = stdout;

static int page_number = 1;
static int page_count = 0;

static int curr_face = -1;

static int horiz_pos = 0;
static int vert_pos = 0;

static int typesetting = 0;


int
DEFUN (main, (arg_count, arg_vec),
int arg_count AND
char **arg_vec)
{
  int i;

  if (arg_vec && arg_vec[0])
    invocation_name = arg_vec[0];

  program_name = strrchr (invocation_name, '/');
  if (program_name)
    ++program_name;
  else
    program_name = invocation_name;

  if (time (&curr_time) == ((time_t) -1))
    die (0);

  strftime (time_string, TIME_SIZE, TIME_FORMAT, localtime (&curr_time));

#ifdef BSD
  siginterrupt (SIGINT, 0);
#else /* not BSD */
  {
    struct sigaction interrupt;

    sigaction (SIGINT, 0, &interrupt);
    interrupt.sa_flags = SA_RESTART;
    sigaction (SIGINT, &interrupt, 0);
  }
#endif /* not BSD */

  set_defaults ();

  get_options (0, 0, "TEXTPS");
  get_options (arg_count, arg_vec, 0);

  if (show_version)
    {
      fprintf (stdout, "textps %s\n", version_string);

      if (show_help == 0)
	exit (EXIT_SUCCESS);
    }

  if (show_help)
    {
      if (show_version)
	fputc ('\n', stdout);

      fputs (help_string, stdout);

      exit (EXIT_SUCCESS);
    }

  if (horiz_offset != 0.0)
    left_margin -= horiz_offset;

  if (vert_offset != 0.0)
    top_margin -= vert_offset;

  if (turn_paper)
    {
      double save;

      save = paper_width;
      paper_width = paper_height;
      paper_height = save;
    }

  arg_count -= optind;
  arg_vec += optind;

  for (i = 0; i < arg_count; ++i)
    {
      if (strcmp (arg_vec[i], "-") == 0)
	continue;

      if (access (arg_vec[i], R_OK) < 0)
	die ("%s:%s: %s\n", program_name, arg_vec[i], strerror (errno));
    }

  if (output_file)
    {
      output_stream = fopen (output_file, "w");
      if (output_stream == 0)
	die ("%s:%s: %s\n", program_name, output_file, strerror (errno));

      output_name = output_file;
    }

  prolog ();

  if (arg_count > 0)
    {
      for (i = 0; i < arg_count; ++i)
	{
	  input_file = arg_vec[i];

	  if (input_stream != stdin)
	    {
	      if (fclose (input_stream) < 0)
		die ("%s:%s: %s\n", program_name, input_name, strerror (errno));
	    }

	  if (strcmp (input_file, "-") == 0)
	    {
	      input_stream = stdin;
	      clearerr (input_stream);

	      input_name = "(standard input)";
	    }
	  else
	    {
	      input_stream = fopen (input_file, "r");
	      if (input_stream == 0)
		die ("%s:%s: %s\n", program_name, input_file, strerror (errno));

	      input_name = input_file;
	    }

	  input_loop ();
	}
    }
  else
    input_loop ();

  trailer ();

  if (input_stream != stdin)
    {
      if (fclose (input_stream) < 0)
	die ("%s:%s: %s\n", program_name, input_name, strerror (errno));
    }

  if (output_stream != stdout)
    {
      if (fclose (output_stream) < 0)
	die ("%s:%s: %s\n", program_name, output_name, strerror (errno));
    }

  exit (EXIT_SUCCESS);
}


static void
DEFUN_VOID (set_defaults)
{
  char *charset;

  charset = getenv ("LC_CTYPE");
  if (charset && *charset)
    {
      if (strcmp (charset, "C") == 0)
	encoding_vec = ascii_encoding;
      else if (strcmp (charset, "POSIX") == 0)
	encoding_vec = ascii_encoding;
      else if (strcmp (charset, "ISO-8859-1") == 0)
	encoding_vec = iso_8859_1_encoding;
    }
}


static void
DEFUN (get_options, (arg_count, arg_vec, env),
int arg_count AND
char **arg_vec AND
const char *env)
{
  int result, option, i;
  char *value, rest;

  if (env)
    {
      value = getenv (env);
      if (value == 0 || value[0] == 0)
	return;

      arg_vec = buildargv (value, invocation_name);
      if (arg_vec == 0)
	die ("%s: Bad environment `%s'\n", program_name, env);

      for (arg_count = 0; arg_vec[arg_count]; ++arg_count)
	;
    }

  for (optind = 0; ; )
    {
      option = getopt_long (arg_count, arg_vec, option_string, option_vec, 0);
      if (option < 0)
	break;

      switch (option)
	{
	case 0:

	  break;

	case MEDIA:

	  for (i = 0; i < media_count; ++i)
	    if (strcasecmp (optarg, media_vec[i].name) == 0)
	      break;

	  if (i == media_count)
	    die ("%s: Unknown paper size `%s'\n", program_name, optarg);

	  paper_width = media_vec[i].width;
	  paper_height = media_vec[i].height;

	  break;

	case PAPER_WIDTH:

	  result = scan_dimen (optarg, &paper_width, 0);
	  if (result != 1 || paper_width <= 0.0)
	    die ("%s: Invalid paper width `%s'\n", program_name, optarg);

	  break;

	case PAPER_HEIGHT:

	  result = scan_dimen (optarg, &paper_height, 0);
	  if (result != 1 || paper_height <= 0.0)
	    die ("%s: Invalid paper height `%s'\n", program_name, optarg);

	  break;

	case ENCODING:

	  for (i = 0; i < charset_count; ++i)
	    if (strcasecmp (optarg, charset_vec[i].name) == 0)
	      break;

	  if (i == charset_count)
	    die ("%s: Unknown character encoding `%s'\n", program_name, optarg);

	  encoding_vec = charset_vec[i].enc;

	  break;

	case FONT_PITCH:

	  result = sscanf (optarg, "%lf%c", &font_pitch, &rest);
	  if (result != 1 || font_pitch <= 0.0)
	    die ("%s: Invalid font pitch `%s'\n", program_name, optarg);

	  char_width = 72.0 / font_pitch;
	  font_size = char_width / 0.6;

	  top_skip = 0.85 * font_size;
	  baseline_skip = 1.2 * font_size;

	  break;

	case FONT_SIZE:

	  result = scan_dimen (optarg, &font_size, 0);
	  if (result != 1 || font_size <= 0.0)
	    die ("%s: Invalid font size `%s'\n", program_name, optarg);

	  char_width = 0.6 * font_size;
	  font_pitch = 72.0 / char_width;

	  top_skip = 0.85 * font_size;
	  baseline_skip = 1.2 * font_size;

	  break;

	case LEFT_MARGIN:

	  result = scan_dimen (optarg, &left_margin, 0);
	  if (result != 1)
	    die ("%s: Invalid left margin `%s'\n", program_name, optarg);

	  break;

	case TOP_MARGIN:

	  result = scan_dimen (optarg, &top_margin, 0);
	  if (result != 1)
	    die ("%s: Invalid top margin `%s'\n", program_name, optarg);

	  break;

	case BIND_OFFSET:

	  result = scan_dimen (optarg, &bind_offset, 0);
	  if (result != 1)
	    die ("%s: Invalid binding offset `%s'\n", program_name, optarg);

	  break;

	case HORIZ_OFFSET:

	  result = scan_dimen (optarg, &horiz_offset, 0);
	  if (result != 1)
	    die ("%s: Invalid horizontal offset `%s'\n", program_name, optarg);

	  break;

	case VERT_OFFSET:

	  result = scan_dimen (optarg, &vert_offset, 0);
	  if (result != 1)
	    die ("%s: Invalid vertical offset `%s'\n", program_name, optarg);

	  break;

	case TOP_SKIP:

	  result = scan_dimen (optarg, &top_skip, 0);
	  if (result != 1)
	    die ("%s: Invalid top skip `%s'\n", program_name, optarg);

	  break;

	case BASELINE_SKIP:

	  result = scan_dimen (optarg, &baseline_skip, 0);
	  if (result != 1)
	    die ("%s: Invalid baseline skip `%s'\n", program_name, optarg);

	  break;

	case LINES_PER_PAGE:

	  result = sscanf (optarg, "%d%c", &lines_per_page, &rest);
	  if (result != 1 || lines_per_page < 0)
	    die ("%s: Invalid number of lines per page `%s'\n", program_name, optarg);

	  break;

	case TAB_WIDTH:

	  result = sscanf (optarg, "%d%c", &tab_width, &rest);
	  if (result != 1 || tab_width < 1)
	    die ("%s: Invalid tab width `%s'\n", program_name, optarg);

	  break;

	case OUTPUT:

	  if (strcmp (optarg, "") == 0)
	    die ("%s: Output file name is empty\n", program_name);

	  output_file = strdup (optarg);
	  if (output_file == 0)
	    die (0);

	  break;

	case FILTER:

	  for (i = 0; optarg[i]; ++i)
	    {
	      if (! isspace (optarg[i]))
		break;
	    }

	  if (optarg[i] == 0)
	    die ("%s: Invalid filter command `%s'\n", program_name, optarg);

	  if (filter_count == filter_size)
	    {
	      int new_size = filter_size + 8;

	      filter_vec = realloc (filter_vec, new_size * sizeof (struct child));
	      if (filter_vec == 0)
		die (0);

	      for (; filter_size < new_size; ++filter_size)
		{
		  filter_vec[filter_size].command = 0;
		  filter_vec[filter_size].arg_vec = 0;
		  filter_vec[filter_size].proc_id = 0;
		}
	    }

	  filter_vec[filter_count].command = strdup (optarg);
	  if (filter_vec[filter_count].command == 0)
	    die (0);

	  filter_vec[filter_count].arg_vec = buildargv (optarg, 0);
	  if (filter_vec[filter_count].arg_vec == 0)
	    die ("%s: Invalid filter command `%s'\n", program_name, optarg);

	  ++filter_count;

	  break;

	case NROFF:

	  font_size = 10.0;
	  font_pitch = 12.0;
	  char_width = 6.0;

	  left_margin = IN (1.0);
	  top_margin = 0.0;

	  top_skip = 8.5;
	  baseline_skip = 12.0;

	  tab_width = 8;
	  lines_per_page = 66;

	  no_bold = 0;
	  no_italic = 0;

	  turn_paper = 0;

	  break;

	default:

	  die ("Try `%s --help' for more information.\n", invocation_name);
	}
    }

  if (env)
    {
      if (optind < arg_count)
	die ("%s: Bad environment `%s'\n", program_name, env);

      freeargv (arg_vec);
    }
}


/* Generate the PostSript header, define the character encoding vector,
   recode and scale the fonts and define some typesetting macros.  */

static void
DEFUN_VOID (prolog)
{
  int i, c;
  char *s;

  fprintf (output_stream, "\
%%!PS-Adobe-2.0
%%%%Title: %s
%%%%Creator: textps %s
%%%%CreationDate: %s
%%%%DocumentFonts: %s %s %s %s
%%%%Pages: (atend)
%%%%PageOrder: Ascend
%%%%Orientation %s
%%%%EndComments\n",
      output_name,
      version_string,
      time_string,
      face_vec[ROMAN],
      face_vec[BOLD],
      face_vec[ITALIC],
      face_vec[BOLD | ITALIC],
      turn_paper ? "Landscape" : "Portrait");

  fputs ("\n/TextEncoding\n", output_stream);

  for (c = i = 0; i < 256; ++i)
    {
      for (; c + strlen (encoding_vec[i]) > 76; c = 0)
	fputc ('\n', output_stream);

      c += fprintf (output_stream, " /%s", encoding_vec[i]);
    }

  fputs ("\n256 packedarray def\n", output_stream);

  fputs ("
/TextRecode {
 dup length dict begin {1 index /FID ne {def} {pop pop} ifelse} forall
 /Encoding TextEncoding def currentdict end definefont pop
} bind def

/TextFont {
 findfont exch scalefont TextRecode
} def\n",
      output_stream);

  s = format_number (0, font_size);

  fprintf (output_stream, "
/TextRoman %s /%s TextFont
/TextBold %s /%s TextFont
/TextItalic %s /%s TextFont
/TextBoldItalic %s /%s TextFont\n",
      s, face_vec[ROMAN],
      s, face_vec[BOLD],
      s, face_vec[ITALIC],
      s, face_vec[BOLD | ITALIC]);

  fprintf (output_stream, "
/%s {/TextRoman findfont setfont} def
/%s {/TextBold findfont setfont} def
/%s {/TextItalic findfont setfont} def
/%s {/TextBoldItalic findfont setfont} def\n",
      face_abbrev[ROMAN],
      face_abbrev[BOLD],
      face_abbrev[ITALIC],
      face_abbrev[BOLD | ITALIC]);

  fputs ("
/M {moveto} def
/S {show} def
%%EndProlog\n",
      output_stream);

  if (ferror (output_stream))
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));
}


/* Finish the PostScript document.  */

static void
DEFUN_VOID (trailer)
{
  fprintf (output_stream, "
%%%%Trailer
%%%%Pages: %d\n", page_count);

  if (control_d)
    fputc (4, output_stream);

  if (ferror (output_stream))
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  if (fflush (output_stream) == EOF)
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));
}


static void
DEFUN_VOID (begin_page)
{
  fprintf (output_stream, "
%%%%Page: %d %d
/TextState save def\n",
      page_number, page_number);

  if (ferror (output_stream))
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  typesetting = 1;
}


static void
DEFUN_VOID (show_page)
{
  fputs ("showpage TextState restore\n", output_stream);

  if (ferror (output_stream))
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  if (fflush (output_stream) == EOF)
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  typesetting = 0;

  ++page_number;
  ++page_count;

  curr_face = -1;
  vert_pos = 0;
}


static void
DEFUN_VOID (input_loop)
{
  int curr_code, face_flag;

  if (fflush (output_stream) == EOF)
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  if (filter_count > 0)			/* Close your eyes...  */
    {
      int input_desc, input_dup;
      int output_desc, output_dup;
      int i;

      input_desc = fileno (input_stream);
      if (input_desc < 0)
	die (0);

      if (input_desc == STDIN_FILENO)
	input_dup = -1;
      else
	{
	  if (private (input_desc) < 0)
	    die (0);

	  input_dup = dup (STDIN_FILENO);
	  if (input_dup < 0)
	    die (0);

	  if (private (input_dup) < 0)
	    die (0);

	  if (dup2 (input_desc, STDIN_FILENO) < 0)
	    die (0);
	}

      output_desc = fileno (output_stream);
      if (output_desc < 0)
	die (0);

      if (output_desc == STDOUT_FILENO)
	output_dup = -1;
      else
	{
	  if (private (output_desc) < 0)
	    die (0);

	  output_dup = dup (STDOUT_FILENO);
	  if (output_dup < 0)
	    die (0);

	  if (private (output_dup) < 0)
	    die (0);

	  if (dup2 (output_desc, STDOUT_FILENO) < 0)
	    die (0);
	}

      for (i = 0; i < filter_count; ++i)
	{
	  int filter_desc[2], stand_out;
	  char **arg_vec;

	  if (pipe (filter_desc) < 0)
	    die (0);

	  if (private (filter_desc[0]) < 0)
	    die (0);

	  if (private (filter_desc[1]) < 0)
	    die (0);

	  stand_out = dup (STDOUT_FILENO);
	  if (stand_out < 0)
	    die (0);

	  if (private (stand_out) < 0)
	    die (0);

	  if (dup2 (filter_desc[1], STDOUT_FILENO) < 0)
	    die (0);

	  filter_vec[i].proc_id = fork ();
	  switch (filter_vec[i].proc_id)
	    {
	    case 0:

	      arg_vec = filter_vec[i].arg_vec;

	      if (arg_vec[0][0] == '/'
		  || (arg_vec[0][0] == '.'
		      && (arg_vec[0][1] == '/'
			  || (arg_vec[0][1] == '.' && arg_vec[0][2] == '/'))))
		execv (arg_vec[0], arg_vec);
	      else
		execvp (arg_vec[0], arg_vec);

	      fprintf (stderr, "%s:%s: %s\n",
		  program_name, arg_vec[0], strerror (errno));

	      _exit (EXIT_FAILURE);

	    case -1:

	      die (0);
	    }

	  if (dup2 (filter_desc[0], STDIN_FILENO) < 0)
	    die (0);

	  if (close (filter_desc[0]) < 0)
	    die (0);

	  if (close (filter_desc[1]) < 0)
	    die (0);

	  if (dup2 (stand_out, STDOUT_FILENO) < 0)
	    die (0);

	  if (close (stand_out) < 0)
	    die (0);
	}

      if (input_dup >= 0)
	{
	  if (dup2 (STDIN_FILENO, input_desc) < 0)
	    die (0);

	  if (dup2 (input_dup, STDIN_FILENO) < 0)
	    die (0);

	  if (close (input_dup) < 0)
	    die (0);
	}

      if (output_dup >= 0)
	{
	  if (dup2 (STDOUT_FILENO, output_desc) < 0)
	    die (0);

	  if (dup2 (output_dup, STDOUT_FILENO) < 0)
	    die (0);

	  if (close (output_dup) < 0)
	    die (0);
	}
    }

  while (1)
    {
      curr_code = fgetc (input_stream);

    test:
      if (curr_code == EOF)
	break;

      if (! typesetting)
	begin_page ();

      switch (curr_code)
	{
	case '\n':
	case '\f':
	case '\v':

	  print_line (1);

	  if (curr_code != '\v')
	    horiz_pos = 0;

	  if (curr_code == '\f' || vert_pos == lines_per_page)
	    show_page ();

	  continue;

	case '\r':

	  horiz_pos = 0;

	  continue;

	case '\b':

	  if (horiz_pos > 0)
	    --horiz_pos;

	  continue;
	}

      if (no_italic == 0 && curr_code == '_')
	{
	  curr_code = fgetc (input_stream);
	  if (curr_code != '\b')
	    {
	      add_char ('_', ROMAN);

	      ++horiz_pos;

	      goto test;
	    }

	  curr_code = fgetc (input_stream);
	  if (curr_code == EOF)
	    {
	      add_char ('_', ROMAN);

	      goto test;
	    }

	  if (curr_code != '\t' && encoding_vec[curr_code][0] == '.')
	    {
	      add_char ('_', ROMAN);

	      goto test;
	    }

	  face_flag = ITALIC;
	}
      else
	face_flag = 0;

      if (curr_code == '\t')
	{
	  int new_pos;

	  new_pos = horiz_pos;

	  new_pos += tab_width;
	  new_pos /= tab_width;
	  new_pos *= tab_width;

	  for (; horiz_pos < new_pos; ++horiz_pos)
	    add_char (' ', ROMAN | face_flag);
	}
      else if (encoding_vec[curr_code][0] != '.')
	{
	  add_char (curr_code, ROMAN | face_flag);

	  ++horiz_pos;
	}
    }

  if (typesetting)
    {
      print_line (0);

      show_page ();
    }

  if (filter_count > 0)			/* Reap all children.  */
    {
      int count, child, status, i;

      for (count = 0; count < filter_count; )
	{
	  child = waitpid (WAIT_ANY, &status, 0);
	  if (child < 0)
	    die (0);

	  for (i = 0; i < filter_count; ++i)
	    {
	      if (filter_vec[i].proc_id != child)
		continue;

	      if (! WIFEXITED (status))
		die ("%s: Child process `%s' terminated by signal\n",
		    program_name, filter_vec[i].arg_vec[0]);

	      if (WEXITSTATUS (status) != EXIT_SUCCESS)
		die ("%s: Child process `%s' exited unsuccessfully\n",
		    program_name, filter_vec[i].arg_vec[0]);

	      ++count;
	    }
	}
    }
}


/* A line of text is represented as an array of boxes.  Each box has a
   character code and font attribute assigned.  The array index is the
   horizontal position of the box.  */

struct box
  {
    int code;
    int face;
    struct box *over;
  };

static struct box *line_vec = 0;
static int line_length = 0;
static int line_size = 0;

static struct box *box_garbage = 0;

static void
DEFUN (add_char, (code, face),
int code AND
int face)
{
  struct box *box;

  if (horiz_pos >= line_size)
    {
      int new_size = line_size + 256;

      line_vec = realloc (line_vec, new_size * sizeof (struct box));
      if (line_vec == 0)
	die (0);

      for (; line_size < new_size; ++line_size)
	{
	  line_vec[line_size].code = 0;
	  line_vec[line_size].face = 0;
	  line_vec[line_size].over = 0;
	}
    }

  if (horiz_pos >= line_length)
    line_length = horiz_pos + 1;

  box = line_vec + horiz_pos;

  if (box->code)			/* An overstrike operation. */
    {
      struct box *end;

      while (box)
	{
	  if (box->code == code && (box->face & face) == face)
	    {
	      if (no_bold == 0)
		box->face |= BOLD;

	      return;
	    }

	  box = box->over;
	}

      if (box_garbage)
	{
	  box = box_garbage;
	  box_garbage = box->over;
	}
      else
	{
	  box = malloc (sizeof (struct box));
	  if (box == 0)
	    die (0);
	}

      end = line_vec + horiz_pos;
      while (end->over)
	end = end->over;

      end->over = box;
    }

  box->code = code;
  box->face = face;
  box->over = 0;
}


static void
DEFUN (print_line, (force),
int force)
{
  struct box *box;
  char x[NUMBER_SIZE], y[NUMBER_SIZE];
  double h, v;
  int i;

  if (line_length == 0)
    {
      if (force)
	++vert_pos;

      return;
    }

  /* Check if the character sequence `_^H_' should be printed as bold
     or italic.  */

  box = line_vec;

  for (i = 0; i < line_length; ++i)
    {
      if (box->code == '_' && (box->face & ITALIC) == ITALIC)
	{
	  struct box *ref = 0;

	  if (i == 0)
	    {
	      if (i + 1 < line_length)
		ref = box + 1;
	    }
	  else if (i + 1 == line_length)
	    {
	      if (i > 0)
		ref = box - 1;
	    }
	  else
	    {
	      ref = box + 1;

	      if (ref->code == ' ')
		ref = box - 1;
	    }

	  if (ref && ref->face == BOLD)
	    box->face = BOLD;
	}

      ++box;
    }

  /* Compute the reference point of the current line and move the cursor
     to that point.  */

  h = left_margin;
  if (page_number & 1)
    h += bind_offset;

  v = paper_height - top_margin - top_skip - vert_pos * baseline_skip;

  format_number (x, h);
  format_number (y, v);

  fprintf (output_stream, "%s %s M\n", x, y);

  /* Output the basic character sequence.  Overstrike characters will be
     handled later.  */

  box = line_vec;

  for (i = 0; i < line_length; )
    {
      if (box->face != curr_face)
	{
	  fputs (face_abbrev[box->face], output_stream);
	  fputc ('\n', output_stream);

	  curr_face = box->face;
	}

      fputc ('(', output_stream);

      for (; i < line_length; ++i)
	{
	  if (box->face != curr_face)
	    break;

	  if (box->code == '(' || box->code == ')' || box->code == '\\')
	    fputc ('\\', output_stream);

	  fputc (box->code, output_stream);

	  ++box;
	}

      fputs (") S\n", output_stream);
    }

  /* Now output all pending overstrike characters.  */

  for (i = 0; i < line_length; ++i)
    {
      box = line_vec[i].over;
      if (box == 0)
	continue;

      format_number (x, h + i * char_width);

      for (; box; box = box->over)
	{
	  fprintf (output_stream, "%s %s M\n", x, y);

	  if (box->face != curr_face)
	    {
	      fputs (face_abbrev[box->face], output_stream);
	      fputc ('\n', output_stream);

	      curr_face = box->face;
	    }

	  fputc ('(', output_stream);

	  if (box->code == '(' || box->code == ')' || box->code == '\\')
	    fputc ('\\', output_stream);

	  fputc (box->code, output_stream);

	  fputs (") S\n", output_stream);
	}
    }

  if (ferror (output_stream))
    die ("%s:%s: %s\n", program_name, output_name, strerror (errno));

  /* Increment the line counter and reset the data structure.  */

  ++vert_pos;

  for (i = 0; i < line_length; ++i)
    {
      box = line_vec + i;

      box->code = 0;
      box->face = 0;

      while (box->over)
	{
	  struct box *tem;

	  tem = box->over;
	  box->over = tem->over;

	  tem->over = box_garbage;
	  box_garbage = tem;
	}
    }

  line_length = 0;
}


static char *
DEFUN (format_number, (string, number),
char *string AND
double number)
{
  static char buffer[NUMBER_SIZE], *end;

  sprintf (buffer, NUMBER_FORMAT, number);

  end = strchr (buffer, 0);
  while (end[-1] == '0')
    --end;

  if (end[-1] == '.')
    --end;

  *end = 0;

  if (string)
    return (strcpy (string, buffer));
  else
    return (buffer);
}


/* Parse a dimension and assign the value (scaled to PostScript units) to
   the variable pointed to by NUMBER.  The return value is the number of
   successful assignments.  */

static int
DEFUN (scan_dimen, (string, number, not_used),
const char *string AND
double *number AND
int not_used)
{
  int result;
  double value;
  char unit[3];
  char rest;

  result = sscanf (string, "%lf%2s%c", &value, unit, &rest);
  if (result == 2)
    {
      if (unit[0] == 'i' && unit[1] == 'n')		/* Inch. */
	value *= 72.0;
      else if (unit[0] == 'c' && unit[1] == 'm')	/* Centimeter. */
	value *= 72.0 / 2.54;
      else if (unit[0] == 'm' && unit[1] == 'm')	/* Millimeter. */
	value *= 72.0 / 25.4;
      else if (unit[0] == 'p' && unit[1] == 't')	/* Point. */
	value *= 72.0 / 72.27;
      else if (unit[0] == 'p' && unit[1] == 'c')	/* Pica. */
	value *= 72.0 / 72.27 * 12.0;
      else if (unit[0] == 's' && unit[1] == 'p')	/* Scaled point. */
	value *= 72.0 / 72.27 / 65536.0;
      else if (unit[0] == 'd' && unit[1] == 'd')	/* Didot point. */
	value *= 72.0 / 72.27 * (1238.0 / 1157.0);
      else if (unit[0] == 'c' && unit[1] == 'c')	/* Cicero. */
	value *= 72.0 / 72.27 * (1238.0 / 1157.0) * 12.0;
      else if (unit[0] != 'b' || unit[1] != 'p')	/* Not big point. */
	die ("%s: Unknown unit `%s'\n", program_name, unit);
    }
  else if (result != 1)
    die ("%s: Bad dimension `%s'\n", program_name, string);

  *number = value;

  return (1);
}


/* Set the `close on execute' flag of a stream.  */

static int
DEFUN (private, (handle),
int handle)
{
  int flags;

  flags = fcntl (handle, F_GETFD, 0);
  if (flags < 0)
    return (flags);
  else
    return (fcntl (handle, F_SETFD, flags | FD_CLOEXEC));
}


/* Print an error message and exit unsuccessfully.  If FORMAT is null then
   a GNU style standard error message according to the current value of the
   `errno' variable will be printed.  */

static void
DEFUN_VAR (die, (format, VA_LIST),
const char *format AND
VA_DECL)
{
  if (format)
    {
      va_list arg_pointer;

      VA_START (arg_pointer, format);
#if HAVE_VPRINTF
      vfprintf (stderr, format, arg_pointer);
#else
      _doprnt (format, arg_poiner, stderr);
#endif
      va_end (arg_pointer);
    }
  else
    fprintf (stderr, "%s: %s\n", program_name, strerror (errno));

  exit (EXIT_FAILURE);
}


/*
 * local variables:
 * truncate-lines: t
 * end:
 */
